package com.peak.common.util;

import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.BiPredicate;
import java.util.function.Consumer;
import java.util.function.Function;

import org.springframework.beans.BeanUtils;

import com.peak.framework.common.page.IQuery;
import com.peak.prd.base.bo.PageBO;
import com.peak.prd.exception.PrdBaseException;

/**
 * 数据类型转换工具类
 * @author zhangdexin
 *
 */
public class ConvertUtil {
	
	/** 转换数据类型，将sourceList转换返回targetClazz类型列表 */
	public static <T> List<T> convert(List<?> sourceList, Class<T> targetClazz) {
		if (sourceList == null) {
			return null;
		}
		
		List<T> newList = new java.util.ArrayList<T>();
		
		for (int i=0; i<sourceList.size(); i++) {
			newList.add(convert(sourceList.get(i), targetClazz));
		}
		
		return newList;
	}
	
	/** 转换数据类型，将sourceObject转换返回targetClazz类型对象 */
	public static <T> T convert(Object sourceObject, Class<T> targetClazz) {
		if (sourceObject == null) {
			return null;
		}
		
		T obj = null;
		try {
			obj = targetClazz.newInstance();
			BeanUtils.copyProperties(sourceObject, obj); // void org.springframework.beans.BeanUtils.copyProperties(Object source, Object target) throws BeansException
			
			// 对PageBO和IQuery做个特殊处理 2022/12/27
			if (sourceObject instanceof PageBO && obj instanceof IQuery) {
				PageBO pageBO = (PageBO)sourceObject;
				IQuery queryObject = (IQuery)obj;
				queryObject.getPage().setPageNumber(pageBO.getPageNumber());
				queryObject.getPage().setPageSize(pageBO.getPageSize());
			}
			
			return obj;
		} catch (InstantiationException e) {
			throw new PrdBaseException(e);
		} catch (IllegalAccessException e) {
			throw new PrdBaseException(e);
		}
	}
	
	/** 根据转换函数，将sourceList逐个应用func函数转换并返回T类型对象列表
	 * @param sourceList 源数据
	 * @param func 转换子（从源对象转换为T类型的对象）
	 * @return 转换后的T类型对象列表
	 */
	public static <T, R> List<R> convert(List<T> sourceList, Function<T, R> func) {
		if (sourceList == null || sourceList.isEmpty()) return null;
		
		List<R> destList = new ArrayList<R>();
		for (int i=0; i<sourceList.size(); i++) {
			destList.add( func.apply(sourceList.get(i)) );
		}
		return destList;
	}
	
	/**
	 * 对象同名属性复制
	 * @param source 源对象
	 * @param target 目标对象
	 * @author weizhenyong
	 * @CreateDate 2021年3月28日
	 *
	 */
	public static void copyProperties(Object source, Object target) {
		if (source != null) {

			BeanUtils.copyProperties(source, target);
			// 对PageBO和IQuery做个特殊处理，复制翻页参数 2022/12/27
			if (source instanceof PageBO && target instanceof IQuery) {
				PageBO pageBO = (PageBO)source;
				IQuery queryObject = (IQuery)target;
				queryObject.getPage().setPageNumber(pageBO.getPageNumber());
				queryObject.getPage().setPageSize(pageBO.getPageSize());
			}
		}
	}
	
	/**
	 * 将list的集合内的对象同名属性复制
	 * @param sourceList 源对象list
	 * @param targetList 目标对象list
	 * @author weizhenyong
	 * @CreateDate 2021年3月28日
	 *
	 */
	public static void copyProperties(List<?> sourceList, List<?> targetList) {
		if (sourceList != null) {
			for (int i=0; i<sourceList.size(); i++) {
				ConvertUtil.copyProperties(sourceList.get(i), targetList.get(i));
			}
		}
	}
	
	/**
	 * 将propertyList中的条目作为属性，逐个赋值到applyList上
	 * @param applyList 对象列表
	 * @param propertyList 属性列表
	 * @param func 操作函数
	 * @author zhangdexin
	 * @CreateDate 2021年3月31日
	 */
	public static <T, P> void setProperties(List<T> applyList, List<P> propertyList, BiConsumer<T, P> func) {
		if (propertyList == null || propertyList.isEmpty()) return;
		if (applyList == null || applyList.isEmpty()) return;
		if (propertyList.size() != applyList.size()) throw new RuntimeException("applyList length must equals propertyList length");
		
		
		for (int i=0; i<propertyList.size(); i++) {
			if (applyList.get(i) == null || propertyList.get(i) == null) {
				continue;
			}
			func.accept(applyList.get(i), propertyList.get(i));
		}
	}

	/**
	 * 将list中的null元素移除，并返回list本身
	 */
	public static <T> List<T> removeNull(List<T> list) {
		if (list == null || list.isEmpty()) return list;
		
		for (int i=list.size()-1; i>=0; i--) {
			if (list.get(i) == null) {
				list.remove(i);
			}
		}
		return list;
	}
	
	/**
	 * 为entity的prop设置value
	 * @param entity 对象实例
	 * @param propName 属性名
	 * @param propValue 属性值
	 * @throws Exception
	 */
	public static void setProperty(Object entity, String propName, Object propValue) {
		if (entity == null || propName == null || propName.isEmpty()) return;
		
		PropertyDescriptor propField = null;
		String originalFieldName = propName;
		try {
			for (int i=0; i<1; i++) {
				Object item = entity;
				
				if (originalFieldName.contains(".")) { // 属性中如果存在.说明是带下一级目录
					String[] fieldArray = originalFieldName.split("\\."); // 拆分属性值获得多级的属性名称
					for (int j = 0; j < fieldArray.length - 1; j++) { // 循环拆分后属性数组获得新的item
						if (item == null) {
							break;
						}
						PropertyDescriptor tempPicField = getPropertyDescriptor(item.getClass(), fieldArray[j]);
						// 将当前的item设置成属性的item值
						item = tempPicField.getReadMethod().invoke(item);
					}
					// 设置picFieldName为拆分后属性数组中最后一个的值
					propName = fieldArray[fieldArray.length - 1];
				}
				
				if (item == null) continue;
				
				if (propField == null) {
					propField = getPropertyDescriptor(item.getClass(), propName);
				}
				
				if (propValue == null) {
					propField.getWriteMethod().invoke(item, propValue);
				}
				else if (Integer.class.getTypeName().equals(propField.getPropertyType().getTypeName())) {
					propField.getWriteMethod().invoke(item, new Integer(propValue.toString())); // propValue类型需要与set方法里的参数类型匹配	
				}
				else if (Long.class.getTypeName().equals(propField.getPropertyType().getTypeName())) {
					propField.getWriteMethod().invoke(item, new Long(propValue.toString())); // propValue类型需要与set方法里的参数类型匹配	
				}
				else if (Double.class.getTypeName().equals(propField.getPropertyType().getTypeName())) {
					propField.getWriteMethod().invoke(item, new Double(propValue.toString())); // propValue类型需要与set方法里的参数类型匹配	
				}
				else {
					propField.getWriteMethod().invoke(item, propValue); // propValue类型需要与set方法里的参数类型匹配	
				}
			}
		}
		catch (Exception ex) {
			throw new RuntimeException(ex);
		}
	}
	
	// 获取bean的getter/setter属性
	private static PropertyDescriptor getPropertyDescriptor(Class clazz, String name) throws IntrospectionException {
		for (PropertyDescriptor propertyDescriptor : Introspector.getBeanInfo(clazz).getPropertyDescriptors()) {
			if (propertyDescriptor.getName().equals(name)) {
				return propertyDescriptor;
			}
		}
		
		return null;
	}
	
	/**
	 * 参照stdList重构todoList，并返回重构后的新列表，新列表与stdList元素个数和顺序完全一致，可能有null元素补位
	 * @param stdList 标准列表
	 * @param todoList 待重构列表
	 * @param matchFunc 匹配函数
	 * @return
	 */
	public static <S, T> List<T> refact(List<S> stdList, List<T> todoList, BiPredicate<S, T> matchFunc) {
		if (stdList == null) return null;
		
		List<T> list = new ArrayList<T>();
		
		for (S std : stdList) {
			if (todoList == null) {
				list.add(null);
				continue;
			}
			
			T foundObj = null;
			for (T t : todoList) {
				if (std == null || t == null) continue;
				if (!matchFunc.test(std, t)) continue;
				foundObj = t; 
				break; // 找到就退出，匹配应该唯一，如果不唯一，就不要用本方法了
			}
			
			list.add(foundObj); // 无论找到与否，都add
		}
		
		return list;
	}
	
	/** 参照stdList重构todoList，并返回重构后的新列表，新列表与stdList元素个数和顺序完全一致，可能有null元素补位
	 * @param stdList 标准列表
	 * @param todoList 待重构列表
	 * @param matchFunc 匹配函数
	 * @param whenNullFunc 当不匹配的时候执行
	 * @return
	 */
	public static <S, T> List<T> refact(List<S> stdList, List<T> todoList, BiPredicate<S, T> matchFunc, Function<S, T> whenNullFunc) {
		if (stdList == null) return null;
		
		List<T> list = new ArrayList<T>();
		
		for (S std : stdList) {
			if (todoList == null) {
				list.add(whenNullFunc.apply(std));
				continue;
			}
			
			T foundObj = null;
			for (T t : todoList) {
				if (!matchFunc.test(std, t)) continue;
				foundObj = t; 
				break; // 找到就退出，匹配应该唯一，如果不唯一，就不要用本方法了
			}
			
			list.add(foundObj == null ? whenNullFunc.apply(std) : foundObj);
		}
		
		return list;
	}
	
	/** 
	 * 合并、排序、分页列表
	 * @param list1
	 * @param list2
	 * @return
	 */
	public static <T> List<T> combine(List<T> list1, List<T> list2, Comparator<T> comparator, int pageNumber, int pageSize) {
		if (list1 == null || list1.isEmpty()) return list2;
		if (list2 == null || list2.isEmpty()) return list1;
		
		// 合并
		List<T> list = new ArrayList<T>(list1.size() + list2.size());
		list.addAll(list1);
		list.addAll(list2);
		
		// 排序
		list.sort(comparator);
		
		// 分页
		int startIndex = (pageNumber-1) * pageSize;
		int endIndex = Math.min(startIndex + pageSize, list.size());
		return list.subList(startIndex, endIndex);
	}
	

	/**
	 * list转map <br />
	 * 使用示例：Map<Long, Lesson> map = ConvertUtil.listToMap(lessons, e->e.getLessonId());
	 * @param list 源对象list 
	 * @param keyfunc lambda表达式：以Lesson举例：e->e.getLessonId()
	 * @return 转换后的map，如果list是null 或list是[],返回{}
	 */
	public static <T,K> Map<K, T> listToMap(List<T> list, Function<T, K> keyfunc) {
		Map<K, T> map = new HashMap<K, T>();
		if (list == null || list.isEmpty()) return map;
		
		for (int i=0; i<list.size(); i++) {
			if (list.get(i) == null) {
				continue;
			}
			map.put(keyfunc.apply(list.get(i)), list.get(i));
		}
		return map;
	}
	
	
	/**
	 * 给sourceList中的属性赋值
	 * @param sourceList 源对象list 
	 * @param setfunc 赋值方法, e -> e.setStatus(1);
	 * @author wzy
	 * @CreateDate 2021年9月19日
	 *
	 */
	public static <T> void setProperties(List<T> sourceList,  Consumer<T> setfunc) {
		if (sourceList == null || sourceList.isEmpty()) return;	
		
		for (int i=0; i<sourceList.size(); i++) {
			setfunc.accept(sourceList.get(i));
		}
	}
	/** 
	 * 根据propertyMap中的对应key的值给sourceList中的对应属性赋值，使用示例参照：{@link com.peak.demo.Demo}的mapSetPropertiesDemo方法
	 * @param sourceList 需要设置属性的list对象
	 * @param propertyMap 属性map对象
	 * @param sourceGetMapKeyFunc sourceList中对象获得map key属性的方法，会根据此key给sourceList中的对象属性赋值
	 * @param sourceSetPropertyFunc sourceList中对象设置属性的方法
	 * @author wzy
	 * @date 2022/8/27 10:59
	 */
	public static <T,R,K,V>  void setProperties(List<T> sourceList, Map<K,V> propertyMap,
												Function<T,R> sourceGetMapKeyFunc, BiConsumer<T,V> sourceSetPropertyFunc) {
		if (ListUtil.isNullorEmpty(sourceList)) {
			return;
		}
		if (propertyMap == null || propertyMap.isEmpty()) {
			return;
		}
		//循环按照propertyMap对应的值给sourceList属性赋值
		for (T list: sourceList) {
			//获得map中属性值
			V obj = propertyMap.get(sourceGetMapKeyFunc.apply(list));
			//map中存在需要的属性值
			if (obj != null) {
				sourceSetPropertyFunc.accept(list,obj);
			}
		}
	}
}
