package com.peak.common.util;

import com.peak.framework.common.date.CalendarUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * 功能描述: 日期处理常用方法
 *
 * @author 魏振永
 * @version 1.0
 * @since 1.0   - 2014-09-04 07:17
 */
public class DateUtil {

    private static final long ONE_MINUTE = 60000L;
    private static final long ONE_HOUR = 3600000L;
    public static final int ONEDAY_SECOND = 86400;
//	private static java.text.DateFormat m_dfDefault = DateFormat.getDateTimeInstance(DateFormat.DEFAULT, DateFormat.MEDIUM, java.util.Locale.CHINA);

    /**
     * 将秒转化为H:MM:SS格式
     *
     * @param sec 秒，不允许为null
     * @return 将秒转化为H:MM:SS格式
     */
    public static String toHMS(long sec) {
        String rtn = "";//返回值
        long hour = sec / (60 * 60);//小时
        long min = (sec % (60 * 60)) / (60);
        long second = (sec % (60));
        rtn = hour + ":";
        if (min < 10) {//长度为一位
            rtn = rtn + "0" + min + ":";
        } else {//长度为两位
            rtn = rtn + min + ":";
        }
        //长度为一位
        if (second < 10) {
            rtn = rtn + "0" + second;
        } else {//长度为两位
            rtn = rtn + second;
        }
        return rtn;
    }

    /**
     * 将秒转化为中文格式的H小时MM分SS秒
     *
     * @param sec 秒，不允许为null
     * @return 将秒转化为  超过1小时显示（ H小时MM分SS秒）
     * 不超过1小时超过1分钟显示（ MM分SS秒）
     * 不超过1分钟显示（SS秒）
     */
    public static String toChineseHMS(long sec) {
    	if(sec <= 0) return "0秒";
    	
        String rtn = "";//返回值
        long hour = sec / (60 * 60);//小时
        long min = (sec % (60 * 60)) / (60);
        long second = (sec % (60));

        if (hour >= 1) { //超过1小时显示小时
            rtn = hour + "小时";
        }

        if (hour >= 1 || min >= 1) { //超过1分钟或者超过了1小时显示分钟
            if (min < 10) {//长度为一位
                rtn = rtn + "0" + min + "分";
            } else {//长度为两位
                rtn = rtn + min + "分";
            }
        }
        //长度为一位
        if (second < 10) {
            rtn = rtn + "0" + second;
        } else {//长度为两位
            rtn = rtn + second;
        }
        return rtn + "秒";
    }

    /**
     * 将秒转化为分钟
     * @param sec 秒，不允许为null
     * @return
     */
    public static Integer toMinute(long sec) {
        Double minute = sec / 60.0;
        return sec % 60 > 30 ? minute.intValue() + 1 : minute.intValue();
    }

    /**
     * 日期增加天数
     *
     * @param date 原始日期
     * @param days 增加的天数，可以带小数，如2.6
     * @return 增加天数后的日期
     */
    public static java.util.Date add(java.util.Date date, double days) {
        try {
            long lDate = date.getTime();
            long lDay = (long) (days * 24 * 3600 * 1000);
            return new java.util.Date(lDate + lDay);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 日期增加月数
     *
     * @param date 原始日期
     * @param days 增加的月数，可以带小数，如2.6
     * @return 增加月数后的日期
     */
    public static Date monthAddNum(Date time, Integer num) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(time);
        calendar.add(Calendar.MONTH, num);
        Date newTime = calendar.getTime();
        return newTime;
    }

    /**
     * 日期增加分钟
     *
     * @param date 原始日期
     * @param days 增加的分钟数，可以带小数，如2.6
     * @return 增加分钟数后的日期
     */
    public static Date minuteAddNum(Date time, Integer num) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(time);
        calendar.add(Calendar.MINUTE, num);
        Date newTime = calendar.getTime();
        return newTime;
    }

    /**
     * 获取年度的第一天
     * @param time
     * @return
     */
    public static Date getYearStart(Date time) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(time);
        calendar.set(Calendar.MONTH, 0);
        calendar.set(Calendar.DATE, 1);
        Date newTime = calendar.getTime();
        return newTime;
    }

    /**
     * 日期增加月数
     *
     * @param date 原始日期
     * @param days 增加的月数，可以带小数，如2.6
     * @return 增加月数后的日期
     */
    public static Date getYearEnd(Date time) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(time);
        calendar.set(Calendar.MONTH, 11);
        calendar.set(Calendar.DATE, 31);
        Date newTime = calendar.getTime();
        return newTime;
    }

    /**
     * 取得应用服务器当前日期
     *
     * @return 当前日期的年月日 时分秒。格式：yyyy-mm-dd hh:mm:ss
     */
    public static String getCurrentFullString() {
        java.util.Date date = new java.util.Date();
        return getFullString(date);

    }

    /**
     * 取得应用服务器当前日期的时分秒
     *
     * @return 当前日期的时分秒。格式：hh:mm:ss
     */
    public static String getCurrentHMS() {
        return getHMS(new java.util.Date());
    }

    /**
     * 取得应用服务器当前日期的年月日
     *
     * @return 当前日期的年月日。格式：yyyy-mm-dd
     */
    public static String getCurrentYMD() {
        return getYMD(new java.util.Date());
    }

    /**
     * 取得日期的年月日 时分秒。格式：yyyy-mm-dd hh:mm:ss
     *
     * @param date 需要转换的日期
     * @return 日期的年月日 时分秒。格式：yyyy-mm-dd hh:mm:ss
     */
    public static String getFullString(java.util.Date date) {
        return getYMD(date) + " " + getHMS(date);

    }

    /**
     * 取得ES日期的年月日 时分秒。格式：yyyy-mm-ddThh:mm:ss
     *
     * @param date 需要转换的日期
     * @return 日期的年月日 时分秒。格式：yyyy-mm-ddThh:mm:ss
     */
    public static String getESFullString(java.util.Date date) {
        return getYMD(date) + "T" + getHMS(date);

    }

    /**
     * 取得日期的时分秒
     *
     * @param date 需要转换的日期
     * @return 取得日期的时分秒。格式：hh:mm:ss
     */
    public static String getHMS(java.util.Date date) {
        try {
//			java.util.Calendar cal = m_dfDefault.getCalendar();
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            int iH = cal.get(java.util.Calendar.HOUR_OF_DAY);
            int iM = cal.get(java.util.Calendar.MINUTE);
            int iS = cal.get(java.util.Calendar.SECOND);
            String str = "";
            str = (iH < 10) ? "0" + iH : "" + iH;
            str += ":" + ((iM < 10) ? "0" + iM : "" + iM);
            str += ":" + ((iS < 10) ? "0" + iS : "" + iS);
            return str;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    /**
     * 取得日期的时分
     *
     * @param date 需要转换的日期
     * @return 取得日期的时分秒。格式：hh:mm
     */
    public static String getHM(java.util.Date date) {
        try {
//			java.util.Calendar cal = m_dfDefault.getCalendar();
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            int iH = cal.get(java.util.Calendar.HOUR_OF_DAY);
            int iM = cal.get(java.util.Calendar.MINUTE);
            String str = "";
            str = (iH < 10) ? "0" + iH : "" + iH;
            str += ":" + ((iM < 10) ? "0" + iM : "" + iM);
            return str;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    public static int getHour(java.util.Date date) {
        try {
//			java.util.Calendar cal = m_dfDefault.getCalendar();
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            int iH = cal.get(java.util.Calendar.HOUR_OF_DAY);
            return iH;
        } catch (Exception e) {
            e.printStackTrace();
            return 0;
        }

    }

    /**
     * 取得日期的年月日
     *
     * @param date 需要转换的日期
     * @return 日期的年月日
     */
    public static Date getYMDdate(java.util.Date date) {
        java.util.Calendar cal = Calendar.getInstance();
        if (date != null) {
            cal.setTime(date);
        }
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTime();
    }


    /**
     * 取得日期的年月日
     *
     * @param date 需要转换的日期
     * @return 日期的年月日。格式：yyyy-mm-dd
     */
    public static String getYMD(java.util.Date date) {
        try {
//			java.util.Calendar cal = m_dfDefault.getCalendar();
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            int iY = cal.get(java.util.Calendar.YEAR);
            int iM = cal.get(java.util.Calendar.MONTH) + 1;
            int iD = cal.get(java.util.Calendar.DATE);
            String str = "";
            if (iY < 50) {
                str = "" + (iY + 2000);
            } else if (iY < 100) {
                str = "" + (iY + 1900);
            } else {
                str = "" + iY;
            }
            str += "-" + ((iM < 10) ? "0" + iM : "" + iM);
            str += "-" + ((iD < 10) ? "0" + iD : "" + iD);
            return str;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    /**
     * 功能：。
     * 创建日期：(2002-8-14 9:14:49)
     * @return java.util.Date 差
     * @param d1 java.util.Date 被减数
     * @param dDay double 减数，带小数的天数，填写正数
     */
    /**
     * 日期减去天数
     *
     * @param date 原始日期
     * @param days 需要减去的天数，填写正数，如：需要减去3天，那么填写3，而不要写-3
     * @return 减去天数后的日期
     */
    public static java.util.Date minus(java.util.Date date, double days) {
        try {
            long lDate = date.getTime();
            long lDay = (long) (days * 24 * 3600 * 1000);
            return new java.util.Date(lDate - lDay);
        } catch (Throwable e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 两个日期相减的天数
     *
     * @param d1 被减数
     * @param d2 减数
     * @return 两个日期相差的天数
     */
    public static double minus(java.util.Date d1, java.util.Date d2) {
        long lDate1 = d1.getTime();
        long lDate2 = d2.getTime();
        long lDay = lDate1 - lDate2;
        double dDay = lDay / 24.0 / 3600.0 / 1000.0;
        return dDay;
    }

    /**
     * 两个日期相减的天数
     *
     * @param d1 被减数
     * @param d2 减数
     * @return 两个日期相差的天数，有小数就多返回1天
     */
    public static int ceilDay(java.util.Date d1, java.util.Date d2) {
        double day = DateUtil.minus(d1, d2);
        return (int) Math.ceil(day);
    }

    /**
     * 两个日期相减的天数
     *
     * @param d1 被减数
     * @param d2 减数
     * @return 两个日期相差的秒数
     */
    public static double minusForSecond(java.util.Date d1, java.util.Date d2) {
        long lDate1 = d1.getTime();
        long lDate2 = d2.getTime();
        long lDay = lDate1 - lDate2;
        double dDay = lDay / 1000.0;
        return dDay;
    }

    /**
     * 两个日期相减的月份
     *
     * @param d1 被减数
     * @param d2 减数
     * @return 两个日期相差的月份
     */
    public static int monthMinus(java.util.Date d1, java.util.Date d2) {
        Calendar c = Calendar.getInstance();
        int result = 0;
        try {
            c.setTime(d1);
            int year1 = c.get(Calendar.YEAR);
            int month1 = c.get(Calendar.MONTH);

            c.setTime(d2);
            int year2 = c.get(Calendar.YEAR);
            int month2 = c.get(Calendar.MONTH);
            if (year1 == year2) {
                result = month1 - month2;
            } else {
                result = 12 * (year1 - year2) + month1 - month2;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    private static java.util.Date replaceDate(String str) {
        String sNew = str.replace('/', '-');
        java.util.Date date = null;
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        try {
            date = sdf.parse(sNew);
            return date;
        } catch (Exception e) {
            try {
                date = sdf.parse(sNew + " 00:00:00");
                return date;
            } catch (Exception ee) {
                ee.printStackTrace();
                return null;
            }

        }
    }

    /**
     * 字符串转换为日期
     *
     * @param dateStr 需要转换的字符串，格式：yyyy-mm-dd or yyyy-mm-dd hh:mm:ss or yyyy/mm/dd or yyyy/mm/dd hh:mm:ss
     * @return 返回日期类型，<br>如输入：2014-3-4 2:3:4,返回2014-03-04 02:03:04
     * <br>如输入：2014-3-4,返回2014-03-04 00:00:00
     */
    public static java.util.Date toDate(String dateStr) {
        try {
            return new java.util.Date((replaceDate(dateStr)).getTime());
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    /**
     * 获取查询条件中起始日期 格式： yyyy-mm-dd 00:00:00.0
     *
     * @param srcDate 源日期
     * @return 转换后的日期，格式yyyy-mm-dd 00:00:00.0
     */
    public static java.util.Date toSDate(Date srcDate) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        try {
            String str = getYMD(srcDate);
            java.util.Date date = null;
            date = sdf.parse(str);
            return date;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }


    /**
     * 获取查询条件中结束日期 格式： yyyy-mm-dd 23:59:59.0
     *
     * @param srcDate 源日期
     * @return 转换后的日期，格式yyyy-mm-dd 23:59:59.0
     */
    public static java.util.Date toEDate(Date srcDate) {
        if (srcDate == null) {
            return srcDate;
        }
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        try {
            String str = DateUtil.getYMD(srcDate);
            java.util.Date date = null;
            date = sdf.parse(str);
            date = CalendarUtils.add(Calendar.SECOND, date, 3600 * 24 - 1);
            return date;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    /**
     * 判断日期是否相等
     *
     * @param d1 日期一
     * @param d2 日期二
     * @return true：相等，false：不相等
     */
    public static Boolean compareDate(Date d1, Date d2) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        String s1 = sdf.format(d1);
        String s2 = sdf.format(d2);
        if (s1.equals(s2)) return true;
        else return false;
    }


    /**
     * 此处插入方法说明。
     * 创建日期：(2002-8-7 11:57:19)
     *
     * @param args java.lang.String[]
     */
    public static void main(String[] args) {

        Date now = new Date();

        Date d1 = minus(now, 2);

        Date d2 = add(now, 1);





    }

    /**
     * 得到某年某周的第一天
     *
     * @param year 年
     * @param week 第几周，第一周是1
     * @return 某年某周的第一天
     */
    public static Date getFirstDayOfWeek(int year, int week) {
        week = week - 1;
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.YEAR, year);
        calendar.set(Calendar.MONTH, Calendar.JANUARY);
        calendar.set(Calendar.DATE, 1);

        Calendar cal = (Calendar) calendar.clone();
        cal.add(Calendar.DATE, week * 7);

        return getFirstDayOfWeek(cal.getTime());
    }

    /**
     * 得到某年某周的最后一天
     *
     * @param year 年
     * @param week 第几周，第一周是1
     * @return 某年某周的最后一天
     */
    public static Date getLastDayOfWeek(int year, int week) {
        week = week - 1;
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.YEAR, year);
        calendar.set(Calendar.MONTH, Calendar.JANUARY);
        calendar.set(Calendar.DATE, 1);
        Calendar cal = (Calendar) calendar.clone();
        cal.add(Calendar.DATE, week * 7);

        return getLastDayOfWeek(cal.getTime());
    }

    /**
     * 取得输入日期所在周的第一天  周一
     *
     * @param date 日期
     * @return 日期所在周的第一天  周一
     */
    public static Date getFirstDayOfChineseWeek(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_WEEK, calendar.getFirstDayOfWeek()); // MONDAY
        return calendar.getTime();
    }

    /**
     * 取得输入日期所在周的最后一天  周日
     *
     * @param date 日期
     * @return 日期所在周的最后一天 周日
     */
    public static Date getLastDayOfChineseWeek(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_WEEK, calendar.getFirstDayOfWeek() + 6); // Sunday
        return calendar.getTime();
    }

    /**
     * 取得输入日期所在周的第一天
     *
     * @param date 日期
     * @return 日期所在周的第一天
     */
    public static Date getFirstDayOfWeek(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.SUNDAY);
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_WEEK, calendar.getFirstDayOfWeek()); // Sunday
        return calendar.getTime();
    }

    /**
     * 取得输入日期所在周的最后一天
     *
     * @param date 日期
     * @return 日期所在周的最后一天
     */
    public static Date getLastDayOfWeek(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setFirstDayOfWeek(Calendar.SUNDAY);
        calendar.setTime(date);
        calendar.set(Calendar.DAY_OF_WEEK, calendar.getFirstDayOfWeek() + 6); // Saturday
        return calendar.getTime();
    }

    /**
     * 取得输入日期所在周的前一周最后一天
     *
     * @param date 日期
     * @return 日期所在周的前一周最后一天
     */
    public static Date getLastDayOfLastWeek(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return getLastDayOfWeek(calendar.get(Calendar.YEAR), calendar
                .get(Calendar.WEEK_OF_YEAR) - 1);
    }

    /**
     * 返回指定日期的月的第一天
     *
     * @return 指定日期的月的第一天
     */
    public static Date getFirstDayOfMonth(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH),
                1);
        return calendar.getTime();
    }

    /**
     * 返回指定年月的月的第一天
     *
     * @param year  指定年
     * @param month 指定月份 ，一月是0
     * @return 指定年月的月的第一天
     */
    public static Date getFirstDayOfMonth(Integer year, Integer month) {
        Calendar calendar = Calendar.getInstance();
        if (year == null) {
            year = calendar.get(Calendar.YEAR);
        }
        if (month == null) {
            month = calendar.get(Calendar.MONTH);
        }
        calendar.set(year, month, 1);
        return calendar.getTime();
    }

    /**
     * 返回指定日期的月的最后一天
     *
     * @param year  指定年
     * @param month 指定月份 ，一月是0
     * @return 指定年月的月的最后一天
     */
    public static Date getLastDayOfMonth(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH),
                1);
        calendar.roll(Calendar.DATE, -1);
        return calendar.getTime();
    }

    /**
     * 返回指定年月的月的最后一天
     *
     * @param year  指定年
     * @param month 指定月份 ，一月是0
     * @return 指定年月的月的最后一天
     */
    public static Date getLastDayOfMonth(Integer year, Integer month) {
        Calendar calendar = Calendar.getInstance();
        if (year == null) {
            year = calendar.get(Calendar.YEAR);
        }
        if (month == null) {
            month = calendar.get(Calendar.MONTH);
        }
        calendar.set(year, month, 1);
        calendar.roll(Calendar.DATE, -1);
        return calendar.getTime();
    }

    /**
     * 返回指定日期的上个月的最后一天
     *
     * @return 指定日期的上个月的最后一天
     */
    public static Date getLastDayOfLastMonth(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(calendar.get(Calendar.YEAR),
                calendar.get(Calendar.MONTH) - 1, 1);
        calendar.roll(Calendar.DATE, -1);
        return calendar.getTime();
    }

    /**
     * 返回指定日期的季的第一天
     *
     * @return 指定日期的季的第一天
     */
    public static Date getFirstDayOfQuarter(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return getFirstDayOfQuarter(calendar.get(Calendar.YEAR),
                getQuarterOfYear(date));
    }

    /**
     * 返回指定年季的季的第一天
     *
     * @param year    年份
     * @param quarter 季度，第一季度是1
     * @return 指定年季的季的第一天
     */
    public static Date getFirstDayOfQuarter(Integer year, Integer quarter) {
        Calendar calendar = Calendar.getInstance();
        Integer month = new Integer(0);
        if (quarter == 1) {
            month = 1 - 1;
        } else if (quarter == 2) {
            month = 4 - 1;
        } else if (quarter == 3) {
            month = 7 - 1;
        } else if (quarter == 4) {
            month = 10 - 1;
        } else {
            month = calendar.get(Calendar.MONTH);
        }
        return getFirstDayOfMonth(year, month);
    }

    /**
     * 返回指定日期的季的最后一天
     *
     * @param year
     * @param quarter
     * @return
     */
    public static Date getLastDayOfQuarter(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return getLastDayOfQuarter(calendar.get(Calendar.YEAR),
                getQuarterOfYear(date));
    }

    /**
     * 返回指定年季的季的最后一天
     *
     * @param year    指定年
     * @param quarter 指定季度，第一季度是1
     * @return 指定年季的季的最后一天
     */
    public static Date getLastDayOfQuarter(Integer year, Integer quarter) {
        Calendar calendar = Calendar.getInstance();
        Integer month = new Integer(0);
        if (quarter == 1) {
            month = 3 - 1;
        } else if (quarter == 2) {
            month = 6 - 1;
        } else if (quarter == 3) {
            month = 9 - 1;
        } else if (quarter == 4) {
            month = 12 - 1;
        } else {
            month = calendar.get(Calendar.MONTH);
        }
        return getLastDayOfMonth(year, month);
    }

    /**
     * 返回指定日期的上一季的最后一天
     *
     * @return 指定日期的上一季的最后一天
     */
    public static Date getLastDayOfLastQuarter(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return getLastDayOfLastQuarter(calendar.get(Calendar.YEAR),
                getQuarterOfYear(date));
    }

    /**
     * 返回指定年季的上一季的最后一天
     *
     * @param year    指定年
     * @param quarter 指定季度
     * @return 指定年季的上一季的最后一天
     */
    public static Date getLastDayOfLastQuarter(Integer year, Integer quarter) {
        Calendar calendar = Calendar.getInstance();
        Integer month = new Integer(0);
        if (quarter == 1) {
            month = 12 - 1;
        } else if (quarter == 2) {
            month = 3 - 1;
        } else if (quarter == 3) {
            month = 6 - 1;
        } else if (quarter == 4) {
            month = 9 - 1;
        } else {
            month = calendar.get(Calendar.MONTH);
        }
        return getLastDayOfMonth(year, month);
    }

    /**
     * 返回指定日期的季度
     *
     * @param date 日期
     * @return 指定日期的季度
     */
    public static int getQuarterOfYear(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return calendar.get(Calendar.MONTH) / 3 + 1;
    }

    /**
     * 返回指定日期的所在月的日
     *
     * @param date 指定的日期
     * @return 指定日期的所在月的日，第一天为1
     * @author weizhenyong
     */
    public static int getDay(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return calendar.get(Calendar.DAY_OF_MONTH);
    }

    /**
     * 返回指定日期的所在月的日
     *
     * @param date 指定的日期
     * @return 指定日期的所在月，一月为1
     * @author weizhenyong
     */
    public static int getMonth(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return calendar.get(Calendar.MONTH) + 1;
    }

    /**
     * 返回指定日期的季度
     *
     * @param date 指定的日期
     * @return 指定日期的季度，第一季度为1
     */
    public static int getQuarter(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return calendar.get(Calendar.MONTH) / 3 + 1;
    }

    /**
     * 返回指定日的年份
     *
     * @param date 日期
     * @return 指定日期的年份
     */
    public static int getYear(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return calendar.get(Calendar.YEAR);
    }

    /**
     * 取得日期的年月
     *
     * @param date 需要转换的日期
     * @return 日期的年月日。格式：yyyymm，如201701
     * @author weizhenyong
     */
    public static String getYM(Date date) {
        try {
//			java.util.Calendar cal = m_dfDefault.getCalendar();
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            int iY = cal.get(java.util.Calendar.YEAR);
            int iM = cal.get(java.util.Calendar.MONTH) + 1;
            String str = "";
            if (iY < 50) {
                str = "" + (iY + 2000);
            } else if (iY < 100) {
                str = "" + (iY + 1900);
            } else {
                str = "" + iY;
            }
            str += ((iM < 10) ? "0" + iM : "" + iM);
            return str;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    /**
     * 获取past天前的日期
     *
     * @author xuning
     * @date 2018年8月31日
     */
    public static Date getPastDate(int past) {
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.DAY_OF_YEAR, calendar.get(Calendar.DAY_OF_YEAR) - past);
        Date today = calendar.getTime();
        return today;
    }

    public static String getFullFormatTime(String datetime) {
        datetime = datetime.replace('/', '-');

        String date = datetime;
        String time = "00:00:00";
        if (datetime.indexOf(" ") != -1) {
            date = datetime.substring(0, datetime.indexOf(" "));
            time = datetime.substring(datetime.indexOf(" ") + 1);
        }
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.YEAR, Integer.parseInt(date.split("-")[0])); // Saturday
        calendar.set(Calendar.MONTH, Integer.parseInt(date.split("-")[1]) - 1); // Saturday
        calendar.set(Calendar.DATE, Integer.parseInt(date.split("-")[2])); // Saturday
        calendar.set(Calendar.HOUR_OF_DAY, Integer.parseInt(time.split(":")[0])); // Saturday
        calendar.set(Calendar.MINUTE, Integer.parseInt(time.split(":")[1])); // Saturday
        calendar.set(Calendar.SECOND, Integer.parseInt(time.split(":")[2])); // Saturday
        Calendar cal = (Calendar) calendar.clone();
        return getFullString(cal.getTime());
    }


    /**
     * 判断当前时间是否在[startTime, endTime]区间，注意时间格式要一致
     *
     * @param nowTime   当前时间
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @return
     * @author
     */
    public static boolean isEffectiveDate(Date nowTime, Date startTime, Date endTime) {
        if (nowTime.getTime() == startTime.getTime()
                || nowTime.getTime() == endTime.getTime()) {
            return true;
        }

        if (null == startTime || null == endTime) {
            return true;
        } else {
            Calendar date = Calendar.getInstance();
            date.setTime(nowTime);

            Calendar begin = Calendar.getInstance();
            begin.setTime(startTime);

            Calendar end = Calendar.getInstance();
            end.setTime(endTime);

            if (date.after(begin) && date.before(end)) {
                return true;
            } else {
                return false;
            }
        }
    }

    /**
     * @param date
     * @return YYYYMMDD
     * @Description： 通过date, 获得年月日字符串 YYYYMMDD
     * @Author：tgl
     * @CreateDate：2015年11月17日 下午4:22:58
     */
    public static String getYMDString(Date date) {
        //SimpleDateFormat format =new SimpleDateFormat("yyMMdd");
        return new SimpleDateFormat("yyyyMMdd").format(date);
    }

    /**
     * 返回上个月的第一天
     *
     * @param date
     * @return
     * @throws Exception
     */
    public static Date getFirstDayOfLastMonth(Date date) throws Exception {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH, -1);
        SimpleDateFormat mFormat = new SimpleDateFormat("yyyy-MM");
        SimpleDateFormat mFormat2 = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
        return mFormat2.parse(mFormat.format(calendar.getTime()) + "-01 00:00:00");
    }

    /**
     * 返回下个月的第一天
     *
     * @param date
     * @return
     * @throws Exception
     */
    public static Date getFirstDayOfNextMonth(Date date) throws Exception {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH, 1);
        SimpleDateFormat mFormat = new SimpleDateFormat("yyyy-MM");
        SimpleDateFormat mFormat2 = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
        return mFormat2.parse(mFormat.format(calendar.getTime()) + "-01 00:00:00");
    }


    /**
     * 返回指定日期的月的第一天 0小时0分0秒
     *
     * @return 指定日期的月的第一天
     */
    public static Date getFirstDayOfCurrentMonth(Date date) throws Exception {
        SimpleDateFormat mFormat = new SimpleDateFormat("yyyy-MM");
        SimpleDateFormat mFormat2 = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
        return mFormat2.parse(mFormat.format(date) + "-01 00:00:00");
    }


    private static String[] parsePatterns = {"yyyy-MM-dd", "yyyy年MM月dd日",
            "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm", "yyyy/MM/dd",
            "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm", "yyyyMMdd"};
    /**尝试转换成日期，支持不同的日期格式**/
	/*public static Date parseDate(String string) {
	    if (string == null) {
	        return null;
	    }
	    try {
	        return org.apache.commons.lang3.time.DateUtils.parseDate(string, parsePatterns);
	    } catch (ParseException e) {
	        return null;
	    }
	}*/

    /**
     * 取得中文格式的年月日
     *
     * @param date 需要转换的日期
     * @return 日期的年月日。格式：yyyy年mm月dd日
     * @author weizhenyong
     */
    public static String getChineseYMD(java.util.Date date) {
        try {
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            int iY = cal.get(java.util.Calendar.YEAR);
            int iM = cal.get(java.util.Calendar.MONTH) + 1;
            int iD = cal.get(java.util.Calendar.DATE);
            String str = "";
            if (iY < 50) {
                str = "" + (iY + 2000);
            } else if (iY < 100) {
                str = "" + (iY + 1900);
            } else {
                str = "" + iY;
            }
            str += "年" + ((iM < 10) ? "0" + iM : "" + iM);
            str += "月" + ((iD < 10) ? "0" + iD : "" + iD);
            str += "日";
            return str;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }

    }

    public static String getTimedifnow(Date date) {
        Calendar cpdate = Calendar.getInstance();
        cpdate.setTime(date);
        // 今天
        Calendar today = Calendar.getInstance();
        // 昨天
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DATE, -1);
        Calendar yesterday = calendar;

        String timedifnow = "";
        if (cpdate.get(Calendar.YEAR) == today.get(Calendar.YEAR) && cpdate.get(Calendar.MONTH) == today.get(Calendar.MONTH) && cpdate.get(Calendar.DAY_OF_MONTH) == today.get(Calendar.DAY_OF_MONTH)) {
            Long timeBetween = today.getTimeInMillis() - cpdate.getTimeInMillis();
            if (timeBetween < ONE_MINUTE) {
                timedifnow = "刚刚";
            } else if (timeBetween < ONE_HOUR) {
                timedifnow = timeBetween / ONE_MINUTE + "分钟前";
            } else {
                timedifnow = timeBetween / ONE_HOUR + "小时前";
            }
        } else if (cpdate.get(Calendar.YEAR) == yesterday.get(Calendar.YEAR) && cpdate.get(Calendar.MONTH) == yesterday.get(Calendar.MONTH) && cpdate.get(Calendar.DAY_OF_MONTH) == yesterday.get(Calendar.DAY_OF_MONTH)) {
            timedifnow = "昨天";
        } else {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            try {
                Long cptime = sdf.parse(sdf.format(date)).getTime();
                Long todaytime = sdf.parse(sdf.format(today.getTime())).getTime();
                Long daysBetween = todaytime - cptime;
                daysBetween = daysBetween / (60L * 60 * 24 * 1000);
                timedifnow = daysBetween + "天前";
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }
        return timedifnow;
    }

    public static String getNewTimedifnow(Date date) {
        Calendar cpdate = Calendar.getInstance();
        cpdate.setTime(date);
        // 今天
        Calendar today = Calendar.getInstance();
        // 昨天
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DATE, -1);
        //Calendar yesterday = calendar;

        String timedifnow = "";
        if (cpdate.get(Calendar.YEAR) == today.get(Calendar.YEAR) && cpdate.get(Calendar.MONTH) == today.get(Calendar.MONTH) && cpdate.get(Calendar.DAY_OF_MONTH) == today.get(Calendar.DAY_OF_MONTH)) {
            Long timeBetween = today.getTimeInMillis() - cpdate.getTimeInMillis();
            if (timeBetween < ONE_MINUTE) {
                timedifnow = "刚刚";
            } else if (timeBetween < ONE_HOUR) {
                timedifnow = timeBetween / ONE_MINUTE + "分钟前";
            } else {
                timedifnow = timeBetween / ONE_HOUR + "小时前";
            }
        } else {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm");
            timedifnow = sdf.format(date);
        }
        return timedifnow;
    }

    /**
     * 将hh:mm:ss字符串转化成秒
     *
     * @param hmsstr hh:mm:ss字符串
     * @return
     */
    public static Long getSecondByHMS(String hmsstr) {
        try {
            String[] hms = hmsstr.split(":");
            Long relust = 0l;
            switch (hms.length) {
                case 1:
                    relust = Long.parseLong(hms[0]);
                    break;
                case 2:
                    relust = Long.parseLong(hms[0]) * 60 + Long.parseLong(hms[1]);
                    break;
                case 3:
                    relust = (Long.parseLong(hms[0]) * 3600) + Long.parseLong(hms[1]) * 60 + Long.parseLong(hms[2]);
                    break;
                default:
                    relust = null;
            }
            return relust;
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 将hh:mm:ss字符串转化成分钟
     *
     * @param hmsstr hh:mm:ss字符串
     * @return
     */
    public static Integer getMinuteByHMS(String hmsstr) {
        try {
            Long relust = getSecondByHMS(hmsstr);
            Double minute = relust / 60.0;
            return relust % 60 > 30 ? minute.intValue() + 1 : minute.intValue();
        } catch (Exception e) {
            return null;
        }
    }
    
    /**
     * 将秒格式化成 HH:mm:ss 格式;不满两位补零
     *
     * @param seconds
     * @return
     */
    public static String getHMSBySecond(int seconds) {
    	int hour = 0;
        int minute = 0;
        int second = 0;
        if (seconds <= 0) {
            return "00:00";
        } else {
            if (seconds >= 3600) {
                hour = seconds / 3600;
                seconds = seconds % 3600;
            }
            
            if (seconds >= 60) {
                minute = seconds / 60;
                second = seconds % 60;
            } else {
            	second = seconds;
            }
            return timeFormat(hour) + ":" + timeFormat(minute) + ":" + timeFormat(second);
        }
    }
    
    private static String timeFormat(int num) {
        String retStr = null;
        if (num >= 0 && num < 10) {
            retStr = "0" + Integer.toString(num);
        } else {
            retStr = "" + num;
        }
        return retStr;
    }

    /**
     * 获得两个年份之间的年份，包含开始和结束的年份
     *
     * @param startDate
     * @param endDate
     * @return
     */
    public static List<Integer> getBetweenYear(Date startDate, Date endDate) {
        try {
            List<Integer> list = new ArrayList<Integer>();
            int startyear = getYear(startDate);
            int endyear = getYear(endDate);
            if (startyear == endyear) {
                list.add(startyear);
            } else if (endyear < startyear) {
                list.add(endyear);
            } else {
                for (int year = startyear; year <= endyear; year++) {
                    list.add(year);
                }
            }
            return list;
        } catch (Exception e) {
            return null;
        }
    }


    /**
     * 获得降序年份的列表
     *
     * @param startyear 初始年份
     * @return
     */
    public static List<Integer> getYearDescList(int startyear) {
        try {
            List<Integer> list = new ArrayList<Integer>();
            int endyear = getYear(new Date());
            if (startyear == endyear) {
                list.add(startyear);
            } else if (endyear < startyear) {
                list.add(endyear);
            } else {
                for (int year = endyear; year >= startyear; year--) {
                    list.add(year);
                }
            }
            return list;
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 获得两个日期之间的所有日期 ,包含开始日期
     *
     * @param startDate
     * @param endDate
     * @return
     */
    public static List<Date> getBetweenDays(Date startDate, Date endDate) {
        List<Date> dateList = new ArrayList<Date>();
        Calendar tempStart = Calendar.getInstance();
        tempStart.setTime(DateUtil.getYMDdate(startDate));
        //tempStart.add(Calendar.DAY_OF_YEAR, 1);
        Calendar tempEnd = Calendar.getInstance();
        tempEnd.setTime(add(DateUtil.getYMDdate(endDate), 1));
        while (tempStart.before(tempEnd)) {
            dateList.add(DateUtil.getYMDdate(tempStart.getTime()));
            tempStart.add(Calendar.DAY_OF_YEAR, 1);
        }
        return dateList;
    }

    /**
     * 判断当前时间的分秒是否在[beginHMStr, endHMStr]区间
     *
     * @param nowDate
     * @param beginHMStr
     * @param endHMStr
     * @return
     */
    public static Boolean isEffectiveHM(Date nowDate, String beginHMStr, String endHMStr) {
        if (nowDate == null || beginHMStr == null || endHMStr == null) return false;
        SimpleDateFormat sdf = new SimpleDateFormat("HH:mm");
        String nowHm = sdf.format(nowDate);
        return beginHMStr.compareTo(nowHm) <= 0 && endHMStr.compareTo(nowHm) >= 0 ? true : false;
    }


    /**
     * 获得当前系统时间到下一天的秒数
     *
     * @return
     */
    public static Long remainderTomorrowSeconds() {
        Long milliSeconds = 0L;
        SimpleDateFormat formate = new SimpleDateFormat("yyyy-MM-dd");
        SimpleDateFormat dfs = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");

        Date now = new Date();
        Long nowMillis = now.getTime();
        String today = formate.format(now);
        try {
            Date end = dfs.parse(today + " 23:59:59.999");
            Long endMillis = end.getTime();
            milliSeconds = endMillis - nowMillis;// 得到两者的毫秒数
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        return milliSeconds * 1000;
    }

    /**
     * 取得日期的年月时分
     *
     * @param date 需要转换的日期
     * @return 日期的年月日。格式：yyyy-mm-dd hh:mm
     */
    public static String getYMDHM(java.util.Date date) {
        return getYMD(date) + " " + getHM(date);

    }

    /**
     * <p>获得两个时间的结束文本</p>
     *
     * @param date
     * @param endDate
     * @return
     * @author: ldc
     * @version: 2022-3-23 18:29:01
     */
    public static String getTimeendnow(Date date, Date endDate) {
        String timeendnow = "";
        int endday = DateUtil.ceilDay(endDate, date);
        if (endday > 1) { //结束时间超过1天
            timeendnow = "还有" + endday + "天结束";
        } else if (endday == 1) { //结束时间是当天
            //结束的秒数
            double second = minusForSecond(endDate, date);
            if (second > 3600) { //超过1小时
                int hour = (int) Math.ceil(second / 3600);
                timeendnow = "还有" + hour + "小时结束";
            } else if (second <= 60) {
                timeendnow = "马上结束";
            } else {
                int minute = (int) Math.ceil(second / 60);
                timeendnow = "还有" + minute + "分钟结束";
            }
        } else {
            timeendnow = "已结束";
        }
        return timeendnow;
    }

    /**
     * 判断指定时间是否在[startDate, endDate]区间范围内
     *
     * @param originDate
     * @param startDate  范围开始日期，如果为null，表示无限小
     * @param endDate    范围结束日期，如果为null，表示无限大
     * @return boolean true：在范围内，false：不在范围内
     * @author wzy
     * @date 2022/9/11 23:47
     */
    public static boolean isContainDate(Date originDate, Date startDate, Date endDate) {
        if (originDate == null) {
            return false;
        }

        boolean start;//记录是否在范围开始日期内，true：在，false：不在
        boolean end;//记录是否在范围结束日期内，true：在，false：不在
        if (startDate == null) {//表示不限制开始日期
            start = true;
        } else if (originDate.compareTo(startDate) >= 0) { //待判断的日期在开始日期范围内
            start = true;
        } else {//待判断的日期不在开始日期范围内
            start = false;
        }
        //待判断的日期在开始日期范围内，就无需在判断结束日期范围
        if (!start) {
            return false;
        }

        if (endDate == null) {//表示不限制结束日期
            end = true;
        } else if (originDate.compareTo(endDate) <= 0) { //待判断的日期在结束日期范围内
            end = true;
        } else {//待判断的日期不在结束日期范围内
            end = false;
        }
        //因为只有在开始日期范围内即start为true，才会执行到这里，因此直接返回是否再结束日期中就可以了
        return end;
    }
    
    /**
     * <p>根据开始时间结束时间获得业务进行状态</p>
     * @param begindate
     * @param enddate
     * @return
     * @author ldc
     * @version 1.0
     * @date 2022-11-22 11:14:35
     */
    public static int getProgressStatus(Date begindate, Date enddate) {
    	//当前时间
		Date curDate = new Date();
		if (begindate != null && begindate.compareTo(curDate) > 0) { //未开始
			return 0;
		} else if (enddate != null && enddate.compareTo(curDate) < 0) { //已经结束
			return 2;
		} else { //进行中
			return 1;
		}
	}

    
    
    /**
     * 获取当前时间(分钟)
     * @return
     * @throws ParseException 
     */
    public static Date getNowMinute() throws ParseException {
    	 SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:00");
    	 Date parse = sdf.parse(sdf.format(new Date()));
    	 return parse;
    }
    
    /**
     * 获取当前时间(小时)
     * @throws ParseException 
     */
    public static Date  getNowHour() throws ParseException {
    	 SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:00:00");
    	 Date parse = sdf.parse(sdf.format(new Date()));
    	 return parse;
    }
    
    /**
     *  获取当前时间（日期）
     * @throws ParseException 
     */
    public static Date  getNowDay() throws ParseException {
   	   SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
   	   Date parse = sdf.parse(sdf.format(new Date()));
   	   return parse;
   }
}
