package com.peak.common.util;

import com.peak.prd.base.model.ResponseStatusEnum;
import com.peak.prd.base.service.IOssService;
import com.peak.prd.base.util.ExceptionUtil;
import com.peak.prd.config.AppCommonConfig;

import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import lombok.extern.slf4j.Slf4j;

/**
 * 文件下载服务类
 * @author wzy
 * @date 2023/3/10 9:44
 */
@Slf4j
public class DownloadFileUtil {

    /**
     * 下载临时文件，每天晚上系统删除前天生成的文件
     * @param downloadfilename 需要下载的文件，需要传入相对于temp目录的路径，如：excel/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.xls
     * @param filenameToClient 下载文件客户端文件名称，如果不填，默认为downloadfilename文件名
     * @param response HttpServletResponse
     * @author wzy
     * @date 2023/3/10 11:29
     */
    public static void downloadTempFile(String downloadfilename, String filenameToClient, HttpServletResponse response) throws Exception {
        downloadFile(downloadfilename,filenameToClient,"temp",response);
    }

    /**
     * 下载Resource文件
     * @param downloadfilename 需要下载的文件，需要传入相对于resource目录的路径，如：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.png
     * @param filenameToClient 下载文件客户端文件名称，如果不填，默认为downloadfilename文件名
     * @param response
     * @author wzy
     * @date 2023/3/10 11:29
     */
    public static void downloadResourceFile(String downloadfilename, String filenameToClient, HttpServletResponse response) throws Exception {
        downloadFile(downloadfilename,filenameToClient,"resource",response);
    }
    /**
     * 下载classpath下的文件
     * @param downloadfilename 需要下载的文件，需要传入相对于classpaht目录的路径，如：files/importtpl/users.xlsx
     * @param filenameToClient 下载文件客户端文件名称，如果不填，默认为downloadfilename文件名
     * @param response
     * @author wzy
     * @date 2023/5/16 11:05
     */
    public static void downloadClassPathFile(String downloadfilename, String filenameToClient, HttpServletResponse response) throws Exception {
        downloadFile(downloadfilename,filenameToClient,"classpath",response);
    }



    /**
     * 通过流的方式在线打开资源文件
     * @param filename 需要打开的文件，需要传入相对于resource目录的路径，如：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.png
     * @param response
     * @author wzy
     * @date 2023/3/10 11:29
     */
    public static void openResourceFileStream(String filename, HttpServletResponse response) throws Exception {
        if (StringUtil.isNullorEmptyStr(filename) || response == null ) {
            ExceptionUtil.handle(ResponseStatusEnum.FILE_DOWNLOAD_FAILD);
        }
        //1. 安全性检查
        securityCheck(filename);
        //2.判断是否采用oss还是本地文件
        InputStream inputStream = null;
        OutputStream outputStream = null;

        try {
            AppCommonConfig config = SpringUtil.getBean(AppCommonConfig.class);
            if(config.getOssEnable() || filename.startsWith("http")) {//使用OSS或者从网络获取文件
                String strurl ="";
                if (config.getOssEnable()) {//使用oss存储
                    IOssService ossService = (IOssService)SpringUtil.getBean(config.getOssServiceName());
                    if(ossService!=null) {
                        strurl = ossService.getDownloadUrl(filename);
                    }

                } else {//从网络获取文件
                    strurl = filename;
                }
                try {
                    URL url = new URL(strurl);
                    HttpURLConnection httpUrl = (HttpURLConnection) url.openConnection();
                    httpUrl.connect();
                    //获取网络输入流
                    inputStream = httpUrl.getInputStream();
                } catch (Exception e) {
                    log.error("strurl======openResourceFileStream========================"+strurl);
                    throw e;
                }

            } else {//使用本地文件
                //2.检查文件是否存在，如果存在获得文件流
                String rootpath = AppPathUtil.getResourceLocalPath();
                String fullfilename = rootpath + filename;
                File file = new File(fullfilename);
                if (!file.exists()) {
                    log.error("fullfilename======openResourceFileStream========================"+fullfilename);
                    ExceptionUtil.handle(ResponseStatusEnum.FILE_DOWNLOAD_FAILD);
                }
                inputStream = new FileInputStream(file);
            }


            //设置文件类型
            String contenttype = "application/"+ FileUtil.getFileExtention(filename).substring(1).toLowerCase();
            response.setContentType(contenttype);
            response.setCharacterEncoding("UTF-8");
            outputStream = response.getOutputStream();
            byte[] b = new byte[1024 * 10];
            int len  = 0;
            while((len  = inputStream.read(b)) > 0){
                outputStream.write(b, 0, len);
            }
            outputStream.flush();
        }
        finally {
            if (outputStream != null) {
                outputStream.close();
            }
            if (inputStream != null) {
                inputStream.close();
            }
        }
    }


    /**
     * 下载文件
     * @param downloadfilename 需要下载的文件，需要传入相对于resource目录的路径，如：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.png
     * @param filenameToClient 下载文件客户端文件名称，如果不填，默认为downloadfilename文件名
     * @param response
     * @author wzy
     * @date 2023/3/10 11:29
     */
    private static void downloadFile(String downloadfilename, String filenameToClient, String filetype, HttpServletResponse response) throws Exception {
        if (StringUtil.isNullorEmptyStr(downloadfilename) || response == null ) {
            ExceptionUtil.handle(ResponseStatusEnum.FILE_DOWNLOAD_FAILD);
        }
        //1. 安全性检查
        securityCheck(downloadfilename);

        InputStream inputStream = null;

        //客户端名称设置
        if (StringUtil.isNullorEmptyStr(filenameToClient)) {
            filenameToClient = FileUtil.getFilename(downloadfilename);
        }
        //从classpath中下载文件
        if (filetype.equals("classpath")) {
            inputStream = DownloadFileUtil.class.getClassLoader().getResourceAsStream(downloadfilename);
            //文件不存在
            if (inputStream ==  null) {
                log.error("downloadfilename=============downloadFile================="+downloadfilename);
                ExceptionUtil.handle(ResponseStatusEnum.FILE_DOWNLOAD_FAILD);
            }
        } else {//下载临时、资源文件

            AppCommonConfig config = SpringUtil.getBean(AppCommonConfig.class);
            //下载临时文件或者不启用oss存储
            //if (filetype.equals("temp") || !config.getOssEnable()) {
            //因为集群导致导出等生成的临时文件和下载会命中不同的机器，因此如果启用oss，统一从oss下载
            if (!config.getOssEnable()) {
                String rootpath = AppPathUtil.getTempLocalPath();
                if (filetype.equals("resource")) {
                    rootpath = AppPathUtil.getResourceLocalPath();
                }
                //检查文件是否存在
                String fullfilename = rootpath + downloadfilename;
                File file = new File(fullfilename);
                if (!file.exists()) {
                    log.error("fullfilename=============downloadFile================="+fullfilename);
                    ExceptionUtil.handle(ResponseStatusEnum.FILE_DOWNLOAD_FAILD);
                }

                //读取文件流
                inputStream = new FileInputStream(file);
            } else {

                    //下载资源文件且使用OSS存储
                    IOssService ossService = (IOssService) SpringUtil.getBean(config.getOssServiceName());
                    String ossurl = ossService.getDownloadUrl(downloadfilename);
                    if (log.isDebugEnabled()) {
                        log.debug("ossurl========debug=====downloadFile================="+ossurl);
                    }
                    try {
                        URL url = new URL(ossurl);
                        HttpURLConnection httpUrl = (HttpURLConnection) url.openConnection();
                        httpUrl.connect();
                        //获取网络输入流
                        inputStream = httpUrl.getInputStream();
                    } catch (Exception e) {
                        log.error("ossurl=============downloadFile================="+ossurl,e);
                        throw e;
                    }

            }
        }

        OutputStream outputStream = null;
        try {

            //设置响应
            response.setContentType("application/octet-stream;charset=UTF-8");
            //设置头信息和下载文件名称，编码防止中文乱码
            response.setHeader("Content-Disposition","attachment;filename="+ URLEncoder.encode(filenameToClient, "UTF-8"));
            response.setCharacterEncoding("UTF-8");
            outputStream = response.getOutputStream();
            byte[] b = new byte[1024 * 10];
            int len  = 0;
            while((len  = inputStream.read(b)) > 0){
                outputStream.write(b, 0, len);
            }
            outputStream.flush();
        }
        finally {
            if (outputStream != null) {
                outputStream.close();
            }
            if (inputStream != null) {
                inputStream.close();
            }
        }
    }

    /**
     * 校验文件下载是否允许
     * @param downloadfilename 需要下载的文件
     * @author wzy
     * @date 2023/5/16 11:14
     */
    private static void securityCheck(String downloadfilename) {
        if (StringUtil.isNullorEmptyStr(downloadfilename) ) {
            ExceptionUtil.handle(ResponseStatusEnum.FILE_DOWNLOAD_FAILD);
        }
//      文件下载漏洞，不允许带../../下载
        if (downloadfilename.indexOf("..") > -1) {
            ExceptionUtil.handle(ResponseStatusEnum.FILE_DOWNLOAD_FAILD);
        }

        //不允许下载白名单之外的文件类型
        if (!SecurityUtil.isAllowFileext(downloadfilename)) {
            ExceptionUtil.handle(ResponseStatusEnum.FILE_EXT_ERROR);
        }
    }


}
