package com.peak.common.util;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;


public class FileUtil {

	/** 清空目录，不删除目录本身 */
	public static void clearDir(String path) {
		File file = new File(path);
		if (!file.exists()) { // 判断是否待删除目录是否存在
			return;
		}

		String[] contents = file.list(); // 取得当前目录下所有文件和文件夹
		if (contents == null || contents.length == 0) {
			return;
		}
		for (String name : contents) {
			File temp = new File(path, name);
			if (temp.isDirectory()) { // 判断是否是目录
				clearDir(temp.getAbsolutePath()); // 递归调用，删除目录里的内容
				temp.delete(); // 删除空目录
			} 
			else {
				if (!temp.delete()) {// 直接删除文件

				}
			}
		}
	}
	
	/**
	 * 获得文件名的扩展名
	 * @param filename 文件名
	 * @return 返回文件扩展名,包含.，如果没有.返回""，如果filename为null，返回null
	 * @author wzy
	 * @date 2021年12月4日
	 *
	 */
	public static String getFileExtention(String filename) {
		if (filename == null) {
			return null;
		}
		int pos = filename.lastIndexOf(".");
//		不存在.
		if (pos == -1) {
			return "";
		}
		String fileExtention = filename.substring(pos).toLowerCase();
		//检查是否带是带参数，如果是需要将参数截取掉
		int paramPos = fileExtention.lastIndexOf("?");
		if (paramPos == -1) {
			return fileExtention;
		} else {
			return fileExtention.substring(0,paramPos).toLowerCase();
		}

	}

	/**
	 * 查找课程入口文件在path下的首次出现的位置，返回文件在path下不包含path的相对路径，如果未找到返回null<br/>
	 * 查找时，如果子目录下多余一个目录或文件时，将不再向下查找，因为如果下级同时出现存在两个文件时无法判断以哪个为准<br/>
	 * 示例：path=/home/course filename=index.html 文件在 /home/course/computer/idex.html下<br/>
	 * 返回：computer/index.html
	 * @param path 查找文件的路径
	 * @param filename 需要查找的文件名
	 * @return 文件在path下不包含path的相对路径，如果未找到返回null
	 * @author wzy
	 * @date 2022/9/22 14:16
	 */
	public static String  findLessonEnterFile(String path,String filename) {
		return findLessonEnterFile(path,filename,-1);
	}
	/**
	 * 查找课程入口文件在path下的首次出现的位置，返回文件在path下不包含path的相对路径，如果未找到返回null<br/>
	 * 查找时，如果子目录下多余一个目录或文件时，将不再向下查找，因为如果下级同时出现存在两个文件时无法判断以哪个为准<br/>
	 * 最多查找level层级,直属path下算0级
	 * 示例：path=/home/course filename=index.html 文件在 /home/course/computer/idex.html下<br/>
	 * 返回：computer/index.html
	 * @param path 查找文件的路径
	 * @param filename 需要查找的文件名
	 * @param level 查找的层级，层级从1开始，-1表示不限制层级
	 * @return 文件在path下不包含path的相对路径，如果未找到返回null
	 * @author wzy
	 * @date 2022/9/22 14:16
	 */
	public static String  findLessonEnterFile(String path,String filename,Integer level) {
		File file = new File(path);
		if (!file.exists()) { // 目录不存在
			return null;
		}
		String[] fileList = file.list(); // 取得当前目录下所有文件和文件夹
		if (fileList == null || fileList.length == 0) {//目录为空，没有任何子目录和文件
			return null;
		}
		if (fileList.length > 1) {//目录下多余一个目录或文件
			//查看当前目录下是否需要查找的问题
			for (String absfile : fileList) {
				File contentfile = new File(path, absfile);
				//查找时，如果子目录下多余一个目录或文件时，将不再向下查找，因为如果下级同时出现存在两个文件时无法判断以哪个为准
				// 目录无需再递归查找
				if (contentfile.isDirectory()) {
					continue;
				}
				if (absfile.equalsIgnoreCase(filename)) {
					return filename;
				}
			}
			return null;
		} else {//目录下只有一个目录或文件
			File contentfile = new File(path, fileList[0]);
			if (contentfile.isDirectory()) { // 是目录，继续递归查找
				return findLessonEnterFile(contentfile.getAbsolutePath(),fileList[0],filename,level, 1); // 递归调用
			} else {
				if (fileList[0].equalsIgnoreCase(filename)) {
					return filename;
				}
			}
		}
		return null;
	}
	/**
	 * 查找课程入口文件在path下的首次出现的位置，返回文件在path下不包含path的相对路径，如果未找到返回null<br/>
	 * 查找时，如果子目录下多余一个目录或文件时，将不再向下查找，因为如果下级同时出现存在两个文件时无法判断以哪个为准<br/>
	 * 最多查找level层级,直属path下算0级
	 * 示例：path=/home/course filename=index.html 文件在 /home/course/computer/idex.html下<br/>
	 * 返回：computer/index.html
	 * @param path 需要查找的路径
	 * @param relativepath path下遍历到的路径
	 * @param filename 需要查找的文件名
	 * @param originlevel 需要查找的层级
	 * @param curlevel 当前层级
	 * @return 文件在path下不包含path的相对路径，如果未找到返回null
	 * @author wzy
	 * @date 2022/9/23 10:11
	 */
	private static String findLessonEnterFile(String path, String relativepath, String filename, Integer originlevel, Integer curlevel) {
		//限制查找深度大于需要找的深度不再进行查找
		if (originlevel != -1 && originlevel.intValue() < curlevel.intValue()) {
			return null;
		}
		File file = new File(path);
		if (!file.exists()) { // 目录不存在
			return null;
		}
		String[] fileList = file.list(); // 取得当前目录下所有文件和文件夹
		if (fileList == null || fileList.length == 0) {//目录为空，没有任何子目录和文件
			return null;
		}
		if (fileList.length > 1) {//目录下多余一个目录或文件
			//查看当前目录下是否需要查找的问题
			for (String absfile : fileList) {
				File contentfile = new File(path, absfile);
				//查找时，如果子目录下多余一个目录或文件时，将不再向下查找，因为如果下级同时出现存在两个文件时无法判断以哪个为准
				// 目录无需再递归查找
				if (contentfile.isDirectory()) {
					continue;
				}
				if (absfile.equalsIgnoreCase(filename)) {
					return relativepath + "/" +filename;
				}
			}
			return null;
		} else {//目录下只有一个目录或文件
			File contentfile = new File(path, fileList[0]);
			if (contentfile.isDirectory()) { // 是目录，继续递归查找
				relativepath = relativepath + "/" + fileList[0];
				return findLessonEnterFile(contentfile.getAbsolutePath(),relativepath,filename,originlevel,curlevel + 1); // 递归调用
			} else {
				if (fileList[0].equalsIgnoreCase(filename)) {
					return filename;
				}
			}
		}
		return null;
	}

	/**
	 * 查看目标字符串是否在原始字符串结尾
	 * @param originStr 原始字符串
	 * @param desStr 目标字符串
	 * @return true: 在最后；false：不在最后
	 * @author wzy
	 * @date 2022/9/23 9:45
	 */
	public static boolean isEndExists(String originStr, String  desStr) {
		if (originStr == null || desStr == null) {
			return false;
		}
		int pos = originStr.lastIndexOf(desStr);
		//originStr存在desStr且
		return pos != -1 && pos == (originStr.length() - desStr.length());
	}

	/**
	 * <p>复制文件</p>
	 *
	 * @param sourceFile 源文件
	 * @param destFile 目标文件
	 * @since   1.0.0
	 * @author zhk
	 * @date 2022/9/23 0023 13:56 </p>
	 */
	public static boolean copyFile(File sourceFile , File destFile)
			throws FileNotFoundException {
		if(!destFile.exists()){
			destFile.getParentFile().mkdirs();
		}
		InputStream fis = new FileInputStream(sourceFile);
		OutputStream fos = new FileOutputStream(destFile);
		try {
			return copyStream(fis, fos);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			try {
				if(fis!=null) fis.close();
			} catch (Exception e) {
				e.printStackTrace();
			}
			try {
				if(fis!=null) fos.close();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return false;
	}

	/**
	 * <p>复制文件流</p>
	 *
	 * @param in
	 * @param out
	 * @return boolean
	 * @since   1.0.0
	 * @author zhk
	 * @date 2022/9/23 0023 13:57 </p>
	 */
	private  static boolean copyStream(InputStream in, OutputStream out ) {
		try {
			byte[] bytes = new byte[10240];
			int c;
			while ((c = in.read(bytes)) != -1) {
				out.write(bytes, 0, c);
			}
			out.flush();
			return true; // if success then return true
		}catch (Exception e) {
			e.printStackTrace();
		}
		return false;
	}

	/**
	 * <p>写文件内容</p>
	 *
	 * @param file 被写文件
	 * @param str 写入内容
	 * @since   1.0.0
	 * @author zhk
	 * @date 2022/9/23 0023 13:59 </p>
	 */
	public static void writeFile(File file, String str)
	{
		byte[] fileContent=null;
		try {
			fileContent = str.getBytes("UTF-8");
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		if(fileContent!=null){
			writeFile( file,fileContent, false);
		}
	}

	/**
	 * <p>写文件内容</p>
	 *
	 * @param file 被写文件
	 * @param fileContent 写入内容字节
	 * @param append 是否在文件尾部追加
	 * @since   1.0.0
	 * @author zhk
	 * @date 2022/9/23 0023 14:02 </p>
	 */
	public static void writeFile(File file, byte[] fileContent,boolean append)
	{
		if(file.isDirectory()){
			return ;
		}
		FileOutputStream fos=null;
		try {
			fos= new FileOutputStream(file, append);
			fos.write(fileContent);
			fos.flush();
		} catch (Exception e) {
			e.printStackTrace();
		}finally{
			close(fos);
		}
	}

	public static void close(Closeable... closeObjs){
		for(Closeable obj:closeObjs){
			if(obj!=null){
				try{ obj.close();}
				catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * <p>判断文件夹是否存在，不存在新建文件夹</p>
	 * @param filepath 文件夹地址
	 * @return File 文件夹对象
	 * @throws Exception
	 * @since   1.0.0
	 * @author 张恺
	 */
	public static File checkDirExist(String filepath){
		File file=new File(filepath);
		if(!file.exists()){
			file.mkdirs();
		}
		return file;
	}

	public static void main(String[] args) throws Exception {
		String path = "E:\\project\\lms\\temp\\e.xlsx";
//		MultipartFile file = getMultipartFile(path);
//		UploadFileUtil.upload(file,"wzy/",true);
	}
	
	/**
	 * 获得一个指定文件名的存储全路径
	 * @param fileName	指定的文件名
	 * @param sourcePath 指定存储路径
	 * @return
	 * @author xiaqing
	 */
	public static String getPathName(String sourcePath ,String fileName) {
		
		Date curdate = new Date();
		int year = DateUtil.getYear(curdate);
		int month = DateUtil.getMonth(curdate);
		int day = DateUtil.getDay(curdate);
		
		//导出的错误excel文件路径
		String filename = fileName + ".xlsx";
		String excelWritePath = sourcePath +"/" + year + "/" + month + "/" + day + "/";
		
		//导出的错误excel文件全路径,包括共享资源目录的本地路径
		String fullpath = AppPathUtil.getTempLocalPath() + excelWritePath;
		
		//文件夹不存在自动生成
		File destFilePath = new File(fullpath);
		if (!destFilePath.exists()) {
			destFilePath.mkdirs();
		}
		
		return fullpath + filename;
	}


	/** 
	 * 纯文件名，不带路径
	 * @param pathname 含路径的文件名
	 * @return 返回纯文件名，如果pathname为null，返回null
	 * @author wzy
	 * @date 2023/5/16 11:24
	 */
	public static String getFilename(String pathname) {
		if (pathname == null) {
			return null;
		}
		int pos = pathname.lastIndexOf("/");
//		不存在
		if (pos == -1) {
			return pathname;
		}
		return pathname.substring(pos+1);
	}
	/**
	 * 获得UUID文件名，去除了中划线
 	 * @return UUID文件名，去除了中划线
	 * @author wzy
	 * @date 2023/7/30 10:22
	 */
	public static String getUUIDFilename() {
		return UUID.randomUUID().toString().replaceAll("-","");
	}
	/**
	 * 获相对resource目录的保存的文件全路径，以/结尾，不含文件名，示例：coursepic/2021/08/20/
	 * @param saveRelativePath 文件保存在resource下的子目录，以/结尾，示例：coursepic/，必须传参
	 * @return  相对resource目录的保存的文件全路径，以/结尾，不含文件名，示例：coursepic/2021/08/20/
	 * @author wzy
	 * @date 2023/7/30 10:51
	 */
	public static String getSaveFullpath(String saveRelativePath) {
		if (StringUtil.isNullorEmptyStr(saveRelativePath)) {
			return null;
		}
		Date curdate = new Date();
		int year = DateUtil.getYear(curdate);
		int month = DateUtil.getMonth(curdate);
		int day = DateUtil.getDay(curdate);
		return saveRelativePath + year + "/" + month + "/" + day + "/";
	}
	
	/**
	 * <p>获得物理路径下全部文件，包括子目录文件，不包括目录</p>
	 * @param filename
	 * @return
	 * @author ldc
	 * @date   2023年7月31日 上午11:58:08
	 */
	public static List<String> getFilepaths(String filename) {
		try (
			Stream<Path> paths = Files.walk(Paths.get(filename))){
            return paths.filter(Files::isRegularFile)
            		.map(e -> e.toString()).collect(Collectors.toList());
        } catch (Exception e) {
            e.printStackTrace();
        }
		return new ArrayList<>();
	}
	/**
	 * 删除文件
	 * @param filename 全路径文件名
	 * @author wzy
	 * @date 2023/10/24 23:24
	 */
	public static void delFile(String filename) {
		if (StringUtil.isNullorEmptyStr(filename)) {
			return;
		}
		File file = new File(filename);
		if (!file.exists()) {
			return;
		}
		file.delete();
	}


//	/**
//	 * File 转MultipartFile
//	 * @param filename 物理全路径文件名，需要带全部路径
//	 * @return 如果filename==null 返回null，filename不存在返回null
//	 * @author wzy
//	 * @date 2023/7/29 9:35
//	 */
//	public static MultipartFile getMultipartFile(String filename) {
//		if (StringUtil.isNullorEmptyStr(filename)) {
//			return null;
//		}
//		//检查文件是否存在
//		File file = new File(filename);
//		if (!file.exists()) {
//			return null;
//		}
//		return getMultipartFile(file);
//	}
//	/**
//	 * File 转MultipartFile
//	 * @param file
//	 * @return 如果file==null 返回null，file不存在返回null
//	 * @author wzy
//	 * @date 2023/7/29 9:33
//	 */
//	public static MultipartFile getMultipartFile(File file)  {
//		if (file == null) {
//			return null;
//		}
//		if (!file.exists()) {
//			return null;
//		}
//		MultipartFile multipartFile = null;
//		try {
//			FileInputStream fileInputStream = new FileInputStream(file);
//			multipartFile = new MockMultipartFile(file.getName(), file.getName(),
//					ContentType.APPLICATION_OCTET_STREAM.toString(), fileInputStream);
//		} catch (Exception e) {
//			log.error("peaktele===FileUtil=====getMultipartFile======="+e.getMessage());
//			e.printStackTrace();
//		}
//		return multipartFile;
//	}



}
