package com.peak.common.util;


import org.apache.commons.collections.CollectionUtils;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author wzy
 * @date 2022/8/20 9:25
 */
public class ListUtil {
    /**
     * 将list的数据去重并保持原顺序
     *
     * @param sourcelist 源list
     * @return 去重并保持与原来顺序一致的list
     * @author wzy
     * @date 2022/8/20 11:52
     */
    public static <T> List<T> removeDuplicate(List<T> sourcelist) {
        if (ListUtil.isNullorEmpty(sourcelist)) {
            return sourcelist;
        }
        //构造LinkedHashSet去重
        LinkedHashSet<T> linkedHashSet = new LinkedHashSet<>(sourcelist);
        //重新构造为arraylist返回
        return new ArrayList<>(linkedHashSet);
    }

    /**
     * 两个List按照orderList对targetList进行排序，orderList不存在的元素放置在targetList最后面
     *
     * @param targetList 需要排序的list
     * @param orderList  按照此list排序
     * @author wzy
     * @date 2022/8/21 10:21
     */
    public static <T> void sort(List<T> targetList, List<T> orderList) {
        if (ListUtil.isNullorEmpty(targetList) || ListUtil.isNullorEmpty(orderList)) {
            return;
        }
        targetList.sort((o1, o2) -> {
            int o1pos = orderList.indexOf(o1);
            int o2pos = orderList.indexOf(o2);
            //为了将orderList不存在的元素放置在targetList最后面
            if (o1pos != -1) {
                o1pos = targetList.size() - o1pos;
            }
            if (o2pos != -1) {
                o2pos = targetList.size() - o2pos;
            }
            return o2pos - o1pos;
        });
    }

    /**
     * 将list转换为树形结构
     * 树形对象示例：com.peak.prd.appsecurity.resource.vo.ResourceVO
     * 调用方法示例：ListUtil.convertToTree(resourceVOList,0l,ResourceVO::getResourceId,<br/>
     * ResourceVO::getParentId,ResourceVO::getChildren,ResourceVO::setChildren)
     *
     * @param originList      需要转换为树形的原始list，需要按照parentid（层级）、排序号等排好序<br/>
     *                        树形对象可以参照：com.peak.prd.appsecurity.resource.vo.ResourceVO
     * @param rootId          根节点Id，支持rootId=null，parentId=rootId即为根节点
     * @param getIdFunc       获得Id的方法
     * @param getParentIdFunc 获得parentid方法
     * @param getChidernFunc  获得chidern方法
     * @param setChildFunc    设置chidern方法
     * @return 树形结构list
     * @author wzy
     * @date 2022/8/23 10:21
     */
    public static <T> List<T> convertToTree(List<T> originList, Long rootId,
                                            Function<T, Long> getIdFunc,
                                            Function<T, Long> getParentIdFunc,
                                            Function<T, List<T>> getChidernFunc,
                                            BiConsumer<T, List<T>> setChildFunc
    ) {
        //根节点等于null需要特殊处理
        if (rootId == null) {
            return convertToTree(originList, getIdFunc, getParentIdFunc, getChidernFunc, setChildFunc);
        } else {
            List<Long> rootIds = new ArrayList<>();
            rootIds.add(rootId);
            return convertToTree(originList, rootIds, getIdFunc, getParentIdFunc, getChidernFunc, setChildFunc);
        }
//        if (ListUtil.isNullorEmpty(originList)) {
//            return originList;
//        }
//        List<T> rootTree = new ArrayList<>();
//        //将list转为树形list
//        for (T tree : originList) {
//            //0.过滤不符合要求的垃圾数据
//            if (rootId != null && getParentIdFunc.apply(tree) == null) {
//                continue;
//            }
//            //1. 查找所有根节点
//            if (rootId == null && getParentIdFunc.apply(tree) == null) {//根节点为null的情况
//                rootTree.add(tree);
//            } else if (rootId != null && rootId.longValue() == getParentIdFunc.apply(tree).longValue()) {//根节点不为null的情况
//                rootTree.add(tree);
//            }
//
//            //2. 构造节点下的所有子节点
//            for (T node : originList) {
//                //因此再次循环，因此需要再次过滤不符合要求的垃圾数据
//                if (rootId != null && getParentIdFunc.apply(node) == null) {
//                    continue;
//                }
//                //当前tree节点的子节点
//                if (getParentIdFunc.apply(node).equals(getIdFunc.apply(tree))) {
//                    //当前tree节点未添加过孩子节点
//                    if (ListUtil.isNullorEmpty(getChidernFunc.apply(tree))) {
//                        setChildFunc.accept(tree, new ArrayList<>());
//                    }
//                    getChidernFunc.apply(tree).add(node);
//                }
//            }
//        }
//        return rootTree;
    }
    /**
     * 将list转换为树形结构，节点为null的情况
     * 树形对象示例：com.peak.prd.appsecurity.resource.vo.ResourceVO
     * 调用方法示例：ListUtil.convertToTree(resourceVOList,0l,ResourceVO::getResourceId,<br/>
     * ResourceVO::getParentId,ResourceVO::getChildren,ResourceVO::setChildren)
     *
     * @param originList      需要转换为树形的原始list，需要按照parentid（层级）、排序号等排好序<br/>
     *                        树形对象可以参照：com.peak.prd.appsecurity.resource.vo.ResourceVO
     * @param getIdFunc       获得Id的方法
     * @param getParentIdFunc 获得parentid方法
     * @param getChidernFunc  获得chidern方法
     * @param setChildFunc    设置chidern方法
     * @return 树形结构list
     * @author wzy
     * @date 2022/8/23 10:21
     */
    private static <T> List<T> convertToTree(List<T> originList,
                                            Function<T, Long> getIdFunc,
                                            Function<T, Long> getParentIdFunc,
                                            Function<T, List<T>> getChidernFunc,
                                            BiConsumer<T, List<T>> setChildFunc
    ) {
        if (ListUtil.isNullorEmpty(originList)) {
            return originList;
        }
        List<T> rootTree = new ArrayList<>();
        //将list转为树形list
        for (T tree : originList) {
            //1. 查找所有根节点
            if (getParentIdFunc.apply(tree) == null) {//根节点为null的情况
                rootTree.add(tree);
            }

            //2. 构造节点下的所有子节点
            for (T node : originList) {
                //因此再次循环，因此需要再次过滤不符合要求的垃圾数据
                //当前tree节点的子节点
                if (getParentIdFunc.apply(node).equals(getIdFunc.apply(tree))) {
                    //当前tree节点未添加过孩子节点
                    if (ListUtil.isNullorEmpty(getChidernFunc.apply(tree))) {
                        setChildFunc.accept(tree, new ArrayList<>());
                    }
                    getChidernFunc.apply(tree).add(node);
                }
            }
        }
        return rootTree;
    }
    /**
     * 将list转换为树形结构
     * 树形对象示例：com.peak.prd.appsecurity.resource.vo.ResourceVO
     * 调用方法示例：ListUtil.convertToTree(resourceVOList,0l,ResourceVO::getResourceId,<br/>
     * ResourceVO::getParentId,ResourceVO::getChildren,ResourceVO::setChildren)
     *
     * @param originList      需要转换为树形的原始list，需要按照parentid（层级）、排序号等排好序<br/>
     *                        树形对象可以参照：com.peak.prd.appsecurity.resource.vo.ResourceVO
     * @param rootIds         根节点Ids支持多根，支持rootIds=null，parentId=rootId即为根节点
     * @param getIdFunc       获得Id的方法
     * @param getParentIdFunc 获得parentid方法
     * @param getChidernFunc  获得chidern方法
     * @param setChildFunc    设置chidern方法
     * @return 树形结构list
     * @author wzy
     * @date 2022/8/23 10:21
     */
    public static <T> List<T> convertToTree(List<T> originList, List<Long> rootIds,
                                            Function<T, Long> getIdFunc,
                                            Function<T, Long> getParentIdFunc,
                                            Function<T, List<T>> getChidernFunc,
                                            BiConsumer<T, List<T>> setChildFunc
    ) {
        if (ListUtil.isNullorEmpty(originList)) {
            return originList;
        }
        List<T> rootTree = new ArrayList<>();
        //将list转为树形list
        for (T tree : originList) {
            //0.过滤不符合要求的垃圾数据
            if (rootIds != null && getParentIdFunc.apply(tree) == null) {
                continue;
            }
            //1. 查找所有根节点
            if (rootIds == null && getParentIdFunc.apply(tree) == null) {//根节点为null的情况
                rootTree.add(tree);
            } else if (rootIds != null && rootIds.contains(getParentIdFunc.apply(tree))) {//根节点不为null的情况
                rootTree.add(tree);
            }

            //2. 构造节点下的所有子节点
            for (T node : originList) {
                //因此再次循环，因此需要再次过滤不符合要求的垃圾数据
                if (rootIds != null && getParentIdFunc.apply(node) == null) {
                    continue;
                }
                //当前tree节点的子节点
                if (getParentIdFunc.apply(node).equals(getIdFunc.apply(tree))) {
                    //当前tree节点未添加过孩子节点
                    if (ListUtil.isNullorEmpty(getChidernFunc.apply(tree))) {
                        setChildFunc.accept(tree, new ArrayList<>());
                    }
                    getChidernFunc.apply(tree).add(node);
                }
            }
        }
        return rootTree;
    }

    /**
     * 判断list是否为null或size=0
     * @param list 需要判断的list
     * @return boolean 为null或size=0时返回true，否则返回true
     * @author wzy
     * @date 2022/8/22 14:30
     */
    public static boolean isNullorEmpty(List<?> list) {
        return (list == null || list.isEmpty());
    }

    /**
     * 判断list是否不为null或size=0
     * @param list 需要判断的list
     * @return boolean 为null或size=0时返回false，否则返回true
     * @author wzy
     * @date 2022/9/29
     */
    public static boolean notNullorEmpty(List<?> list) {
        return (list != null && !list.isEmpty());
    }

    /**
     * 将list中的元素拼接成string
     * @param list 需要拼接的list
     * @param separator 拼接分隔符
     * @author huangzhiwen
     * @date 10:00 2022/9/27
     **/
    public static String listToString(List<?> list, String separator) {
        if (isNullorEmpty(list)) {
            return null;
        }
        StringBuilder stringBuilder = new StringBuilder();
        list.forEach(e -> stringBuilder.append(e).append(separator));
        return stringBuilder.toString().substring(0, stringBuilder.lastIndexOf(separator));
    }

    /**
     * 去除list中所有的null元素
     * @param list 待去除null的list
     * @author wzy
     * @date 2022/12/4 19:35
     */
    public static <T> void removeAllNull(List<T> list) {
        list.removeAll(Collections.singleton(null));
    }

    /**
     * 差集，listA - listB，原集合不变
     * @param listA 集合A
     * @param listB 集合B
     * @return 一个新的差集集合
     * @author wzy
     * @date 2023/6/22 17:24
     */
    public static <T> List<T> subtract(List<T> listA, List<T> listB) {
        if (listA == null) {
            return new ArrayList<>();
        }
        if (listB == null) {
            return listA;
        }
        return (List<T>)CollectionUtils.subtract(listA,listB);
    }
    /**
     * 并集，原集合不变
     * @param listA 集合A
     * @param listB 集合B
     * @return 一个新的并集集合
     * @author wzy
     * @date 2023/6/22 17:24
     */
    public static <T> List<T> union(List<T> listA, List<T> listB) {
        if (listA == null && listB == null) {
            return new ArrayList<>();
        }
        if (listA == null) {
            return listB;
        }
        if (listB == null) {
            return listA;
        }
        return (List<T>)CollectionUtils.union(listA,listB);
    }

    /**
     * 交集，原集合不变
     * @param listA 集合A
     * @param listB 集合B
     * @return 一个新的交集集合
     * @author wzy
     * @date 2023/6/22 17:24
     */
    public static <T> List<T> intersection(List<T> listA, List<T> listB) {
        if (listA == null || listB == null) {
            return new ArrayList<>();
        }
        return (List<T>)CollectionUtils.intersection(listA,listB);
    }
    
    /**
     * <p>树形列表转换成平铺列表方法</p>
     * @param <T>
     * @param treeList
     * @param getChidernFunc
     * @return
     * @author ldc
     * @date   2023年8月22日 下午3:07:30
     */
    public static <T> List<T> convertTreeToList(List<T> treeList, Function<T, List<T>> getChidernFunc){
		return treeList.stream()
		        .collect(Collectors.toList())
		        .stream()
		        .map(x -> {return flatten(x, getChidernFunc);})
		        .flatMap(Collection::stream)
		        .distinct()
		        .collect(Collectors.toList());
	}
	    
	private static <T> List<T> flatten(T node, Function<T, List<T>> getChidernFunc) {
	    if (node == null) {
	        return null;
	    }
	 
	    //返回的列表
	    List<T> flatList = new ArrayList<>();
	    //子双端队列
    	Deque<T> deque = new ArrayDeque<>();
	    //添加根节点
	    deque.addFirst(node);
	    //保持循环直到遍历所有节点
	    while (!deque.isEmpty()) {
	    	//获得队列中最前面的对象，获得同时在队列中删除
	    	T n = deque.removeLast();
	        flatList.add(n);
	        //获得n的孩子
	        List<T> children = getChidernFunc.apply(n);
	        if (ListUtil.notNullorEmpty(children)) {
	        	//n的孩子存在遍历孩子放到队列最后面
	            for (T child : children) {
	            	deque.addFirst(child);
	            }
	        }
	    }
	    return flatList;
	}
    /**
     * 移除list中对象第一个指定属性值等于removeObj的对象
     * @param list 需要移除的list
     * @param getRemoveObjFunc 获得需要移除属性值方法
     * @param removeObj list中需要移除的属性值
     * @author wzy
     * @date 2023/10/2 12:23
     */
    public static  <T,K> void removeFirst(List<T> list,  Function<T, K> getRemoveObjFunc,K removeObj) {
        for (T t : list) {
            if (getRemoveObjFunc.apply(t).equals(removeObj)) {
                list.remove(t);
                break;
            }
        }
    }

    /**
     * 无序list中的数据有parentid进行关联，按照树结构进行排序，父在前孩子在后
     * @param sourceList      带排序的原始list
     * @param getIdFunc       获得Id的方法
     * @param getParentIdFunc 获得parentid方法
     * @return 按照树结构进行排序后list，父在前孩子在后
     * @author wzy
     * @date 2023/12/3 16:39
     */
    public static <T> List<T> sortByTree(List<T> sourceList,Function<T, Long> getIdFunc,
                                         Function<T, Long> getParentIdFunc){
        if (ListUtil.isNullorEmpty(sourceList)) {
            return sourceList;
        }
        //根节点
        List<T> rootList = new ArrayList<>();
        //结果list
        List<T> resultList = new ArrayList<>();
        //查找所有的根
        for(T node : sourceList){
            if (isRoot(node,sourceList,getIdFunc,getParentIdFunc)) {
                rootList.add(node);
            }
        }

        //循环添加每个root的孩子节点
        for (T root : rootList){
            //将根加到结果节点
            resultList.add(root);
            //将根的孩子及子孩子节点加到根
            addChildren(root,sourceList,resultList,getIdFunc,getParentIdFunc);
        }
        return resultList;
    }
    /**
     * 判断当前节点是否在sourceList为根
     * @param node            当前节点
     * @param sourceList      带排序的原始list
     * @param getIdFunc       获得Id的方法
     * @param getParentIdFunc 获得parentid方法
     * @return boolean true:是根，false：不是根
     * @author wzy
     * @date 2023/12/3 16:43
     */
    private static <T> boolean isRoot(T node,List<T> sourceList,Function<T, Long> getIdFunc,
                                      Function<T, Long> getParentIdFunc){
        //没有父节点就是根节点
        if (getParentIdFunc.apply(node) == null) {
            return true;
        }
        for(T t : sourceList){
            //父Id找到对应的节点就表明不是根
            if (getParentIdFunc.apply(node).equals(getIdFunc.apply(t))){
                return false;
            }
        }
        //父Id没找到对应的节点表明是根
        return true;

    }

    /**
     * 给当前节点添加孩子节点
     * @param node             当前节点
     * @param sourceList      带排序的原始list
     * @param resultList      结果list
     * @param getIdFunc       获得Id的方法
     * @param getParentIdFunc 获得parentid方法
     * @author wzy
     * @date 2023/12/3 16:48
     */
    private static <T> void addChildren(T node,List<T> sourceList,List<T> resultList,Function<T, Long> getIdFunc,
                              Function<T, Long> getParentIdFunc){
        List<T> childrenList = new ArrayList<>();
        //遍历输入列表，添加孩子节点
        for(T t : sourceList){
            if(getIdFunc.apply(node).equals(getParentIdFunc.apply(t))) {
                childrenList.add(t);
            }
        }
        //遍历到最末端，无下级，退出遍历
        if(childrenList.isEmpty()){
            return;
        }
        //递归遍历子孩子
        for(T t : childrenList){
            resultList.add(t);
            addChildren(t,sourceList,resultList,getIdFunc,getParentIdFunc);
        }
    }


    public static void main(String[] args) {
        List<Integer> targetList = new ArrayList<>();
        targetList.add(222);
        targetList.add(111);
        List<Integer> twoList = new ArrayList<>();
        twoList.add(222);
        twoList.add(333);

        twoList.retainAll(targetList);


    }
}
