package com.peak.common.util;

import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import com.peak.prd.config.RedisConfig;

@Component
public class RedisObjectService {
	@Autowired
	private RedisTemplate<String, Object> redisTemplate;
	
	@Autowired
	RedisConfig redisConfig;
	
	// 附加前缀
	String getKey(String key) {
		String prefix = redisConfig.getKeyPrefix();
		if (prefix == null) return key;
		
		return prefix + "::" + key;
	}
	
	List<String> getKeys(List<String> keys) {
		String prefix = redisConfig.getKeyPrefix();
		if (prefix == null || keys == null) return keys;
		
		List<String> newKeys = new java.util.ArrayList<String>();
		for (int i=0; i<keys.size(); i++) {
			newKeys.add(prefix + "::" + keys.get(i));
		}
		
		return newKeys;
	}
	
	/**
	 * mget批量查询
	 * @param keys 允许有重复key
	 * @return 和keys的数量及顺序完全一致(某些项可能为null)
	 */
	public List<Object> multiGet(List<String> keys) {
		return redisTemplate.opsForValue().multiGet( this.getKeys(keys) );
	}
	
	/**
	 * GET key，返回 key所关联的字符串值。
	 * @param key
	 * @return value
	 */
	public Object get(String key) {
		return redisTemplate.opsForValue().get( this.getKey(key) );
	}
	
	/**
	 * SET key value，设置一个key-value
	 * @param key
	 * @param value
	 */
	public void set(String key, Object value) {
		redisTemplate.opsForValue().set( this.getKey(key) , value);
	}
	
	/**
	 * 实现命令：SET key value EX seconds，设置key-value和超时时间（秒）
	 * 
	 * @param key
	 * @param value
	 * @param secondsTimeout        （以秒为单位）
	 */
	public void set(String key, Object value, long secondsTimeout) {
		redisTemplate.opsForValue().set( this.getKey(key) , value, secondsTimeout, TimeUnit.SECONDS);
	}
	
	/**
	 * 如果key不存在，则设置  SET if Not eXists
	 * @param key
	 * @param value
	 */
	public Boolean setnx(String key, Object value) {
		return redisTemplate.opsForValue().setIfAbsent( this.getKey(key) , value);
	}
	
	/**
	 * 实现命令：expire 设置过期时间，单位秒
	 * 
	 * @param key
	 * @return
	 */
	public void expire(String key, long timeout) {
		redisTemplate.expire( this.getKey(key) , timeout, TimeUnit.SECONDS);
	}
	
	/**
	 * 实现命令：DEL key，删除一个key
	 * 
	 * @param key
	 */
	public void del(String key) {
		redisTemplate.delete( this.getKey(key) );
	}
	
	/**
	 * 判断key是否存在
	 * @param key
	 * @return
	 */
	public boolean hasKey(String key) {
		if (key == null || key.isEmpty()) {
			return false;
		}
		
		return redisTemplate.hasKey( this.getKey(key) );
	}

	/**
	 * 放哈希key-value对
	 * @param key
	 * @param hashKey
	 * @param hashValue
	 */
	public void putHash(String key, Object hashKey, Object hashValue) {
		redisTemplate.opsForHash().put( this.getKey(key) , hashKey, hashValue);
	}
	
	/**
	 * 获取哈希值
	 * @param key
	 * @param hashKey
	 * @return
	 */
	public Object getHashValue(String key, Object hashKey) {
		return redisTemplate.opsForHash().get( this.getKey(key) , hashKey);
	}
	
	/**
	 * 获取哈希keys
	 * @param key
	 * @return
	 */
	public Set<Object> getHashKeys(String key) {
		return redisTemplate.opsForHash().keys( this.getKey(key) );
	}
	
	/**
	 * 删除一些哈希
	 * @param key
	 * @param hashKeys
	 */
	public void delHash(String key, Object... hashKeys) {
		redisTemplate.opsForHash().delete( this.getKey(key) , hashKeys);
	}
}
