package com.peak.common.util;

import com.peak.prd.config.RedisConfig;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.Cursor;
import org.springframework.data.redis.core.ScanOptions;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.core.script.DefaultRedisScript;
import org.springframework.stereotype.Component;

@Slf4j
@Component
public class RedisService {
	@Autowired
	private StringRedisTemplate redisTemplate;
	
	@Autowired
	private RedisConfig redisConfig;
	
	// 附加前缀
	public String getKey(String key) {
		String prefix = redisConfig.getKeyPrefix();
		if (prefix == null) {
			return key;
		}
		
		return prefix + "::" + key;
	}
	
	public List<String> getKeys(List<String> keys) {
		String prefix = redisConfig.getKeyPrefix();
		if (prefix == null || keys == null) {
			return keys;
		}
		
		List<String> newKeys = new java.util.ArrayList<String>();
		for (int i=0; i<keys.size(); i++) {
			newKeys.add(prefix + "::" + keys.get(i));
		}
		
		return newKeys;
	}
	
	/**
	 * redis incr命令
	 * @param key
	 * @param delta 增加步长，一般为1
	 * @return 增加后的值
	 */
	public long increment(String key, long delta) {
		return redisTemplate.opsForValue().increment( this.getKey(key) , delta);
	}
	
	/**
	 * <p>TODO</p>
	 * @param key 有序队列key
	 * @param value 元素名称
	 * @param delta 给元素增加步长，一般为1
	 * @return Double
	 * @since   1.0.0
	 * @author zhk
	 * @date 2022年1月21日 下午2:56:27 </p>
	 */
	public Double incrementZSetScore(String key, String value, long delta) {
		return redisTemplate.opsForZSet().incrementScore( this.getKey(key) , value, delta);
	}
	
	public Long getZSetrank(String key, String value) {
		return redisTemplate.opsForZSet().rank( this.getKey(key) , value);
	}
	
	
	public Long getZSetReverseRank(String key, String value) {
		return redisTemplate.opsForZSet().reverseRank( this.getKey(key) , value);
	}

	public Double getZSetScore(String key, String value) {
		return redisTemplate.opsForZSet().score( this.getKey(key) , value);
	}
	
	public Long getZSetSize(String key) {
		return redisTemplate.opsForZSet().size( this.getKey(key) );//等同于zCard(key);
	}

	/**
	 * mget批量查询
	 * @param keys 允许有重复key
	 * @return 和keys的数量及顺序完全一致(某些项可能为null)
	 */
	public List<String> multiGet(List<String> keys) {
		return redisTemplate.opsForValue().multiGet( this.getKeys(keys) );
	}
	
	/**
	 * GET key，返回 key所关联的字符串值。
	 * @param key
	 * @return value
	 */
	public String get(String key) {
		return redisTemplate.opsForValue().get( this.getKey(key) );
	}
	
	/**
	 * SET key value，设置一个key-value
	 * @param key
	 * @param value
	 */
	public void set(String key, String value) {
		redisTemplate.opsForValue().set( this.getKey(key) , value);
	}
	
	/**
	 * 实现命令：SET key value EX seconds，设置key-value和超时时间（秒）
	 * 
	 * @param key
	 * @param value
	 * @param secondsTimeout        （以秒为单位）
	 */
	public void set(String key, String value, long secondsTimeout) {
		redisTemplate.opsForValue().set( this.getKey(key) , value, secondsTimeout, TimeUnit.SECONDS);
	}
	
	/**
	 * 如果key不存在，则设置  SET if Not eXists
	 * @param key
	 * @param value
	 */
	public Boolean setnx(String key, String value) {
		return redisTemplate.opsForValue().setIfAbsent( this.getKey(key) , value);
	}
	
	/**
	 * 如果key不存在，则设置  SET if Not eXists
	 * @param key
	 * @param value
	 */
	public Boolean setnx(String key, String value, long secondsTimeout) {
		return redisTemplate.opsForValue().setIfAbsent( this.getKey(key) , value, secondsTimeout, TimeUnit.SECONDS);
	}
	/**
	 * 设置锁前缀key
	 * @param key
     * @return java.lang.String
	 * @author weizhenyong
	 * @date 2024/7/8 14:02
	 */
	public String getLockKey(String key) {
		return "lock:" + key;
	}
	
	/**
	 *进行加锁
	 * @param key 加锁key，尽量能体现业务含义
	 * @param value 自己设置的一个唯一值，建议使用UUID，用于解锁时使用，<br/>
	 *              作用：防止锁到达secondsTimeout后过期释放了，原请求还没执行完，然后新的请求获取到了锁，
	 *              这时候原请求执行完了，解锁时如果不判断锁中的值是否自己加的，就会将新的请求的锁给解锁了
	 * @param secondsTimeout 设置锁超时时间，正常需要大于锁代码的执行时间，但也不能太大，太大会导致系统堵塞，严重影响系统性能，
	 *                       最大不允许超过1分钟，超过一分钟需要拆分业务代码了
	 *                       作用：设置超时时间是为了防止程序出问题锁不释放
	 * @return true: 加锁成功，false: 加锁失败
	 * @author weizhenyong
	 * @date 2024/7/7 0:14
	 */
	public Boolean lock(String key, String value, long secondsTimeout) {
		if (StringUtil.isNullorEmptyStr(key) || StringUtil.isNullorEmptyStr(value)) {
			return false;
		}
		key = this.getLockKey(key);
		
		return setnx(key,value,secondsTimeout);
	}

	/**
	 * 提供锁名称，获取锁值（null表示未加锁）
	 * @param key 锁名
	 * @return 锁值
	 * @author zhangdexin
	 * @date 2024/9/30
	 */
	public String getLockValue(String key) {
		return this.get(this.getLockKey(key));
	}

	/**
	 * 进行加锁，如果不能获得锁，将进行重试，超过1分钟还未获得锁，将抛出exception
	 * @param key 加锁key，尽量能体现业务含义
	 * @param value 自己设置的一个唯一值，建议使用UUID，用于解锁传入
	 * @param secondsTimeout 设置锁超时时间，正常需要大于锁代码的执行时间，但也不能太大，太大严重影响系统性能，
	 *                       最大不允许超过1分钟，超过一分钟需要拆分业务代码了
	 * @return true: 加锁成功，false: 加锁失败
	 * @author weizhenyong
	 * @date 2024/7/7 0:09
	 */
	public Boolean lockBlockOneMinute(String key, String value, long secondsTimeout) throws Exception {
		if (StringUtil.isNullorEmptyStr(value)) {
			return false;
		}
		long start = System.currentTimeMillis();
		//循环获得锁，超过60秒，则不再等待
		while (true) {
			Boolean lock = lock(key,value,secondsTimeout);
			if (Boolean.TRUE.equals(lock)) {
				return true;
			}
			// 等待超过60秒，则不再等待
			if (System.currentTimeMillis() - start > 60000) {
				log.error("=========peak==============Get lock fail!");
				throw new Exception("key==========" + key + "value=====" + value + "====get lock fail");
            }
			// 等待100毫秒后重试
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				return false;
			}
		}
	}


	/**
	 * 解锁（只能解除自己加的锁）
	 * @param key 加锁时填写的key
	 * @param value 加锁时设置key的值
	 * @author weizhenyong
	 * @date 2024/11/11 13:28
	 */
	public void unlock(String key, String value) {
		key = this.getLockKey(key);
		String oldValue = this.get(key);
		//如果key对应的值发生了改变，表明锁已经释放了
		if (oldValue == null || !oldValue.equals(value)) {
			return;
		}
		this.del(key);
	}

	/**
	 * 强制解锁
	 * @param key 锁名称
	 */
	public void delLock(String key) {
		this.del(this.getLockKey(key));
	}
	
	/**
	 * 实现命令：expire 设置过期时间，单位秒
	 * 
	 * @param key
	 * @return
	 */
	public void expire(String key, long timeout) {
		redisTemplate.expire( this.getKey(key) , timeout, TimeUnit.SECONDS);
	}
	
	/**
	 * 实现命令：DEL key，删除一个key
	 * 
	 * @param key
	 */
	public Boolean del(String key) {
		return redisTemplate.delete( this.getKey(key) );
	}
	
	/**
	 * 判断key是否存在
	 * @param key
	 * @return
	 */
	public boolean hasKey(String key) {
		if (key == null || key.isEmpty()) {
			return false;
		}
		
		return redisTemplate.hasKey( this.getKey(key) );
	}
	
	/**
	 * 重命名redis key
	 * @param oldKey
	 * @param newKey
	 */
	public void rename(String oldKey, String newKey) {
		redisTemplate.rename( this.getKey(oldKey) ,  this.getKey(newKey) );
	}
	
	
	
	/**
	 * san 方法,
	 * 
	 * @param 规则key
	 * @param 一次取多少条
	 * @return
	 */
	/*public Map<String, String> scan(String matchKey, Long count) {
		Map<String, String> retMap = redisTemplate.execute(connection -> {
			Map<String, String> keysTmp = new HashMap<String, String>();
			Cursor<byte[]> cursor = connection
					.scan(new ScanOptions.ScanOptionsBuilder().match("*" + matchKey + "*").count(count).build());
			while (cursor.hasNext()) {
				String key = new String(cursor.next());
				keysTmp.put(key, key);
			}
			return keysTmp;
		}, Boolean.TRUE);
		return retMap;
	}*/

	/**
	 * 获取指定的key有效时间
	 * 
	 * @param key
	 * @param timeunit
	 * @return
	 */
	public long getExpire(String key, TimeUnit timeunit) {
		if (timeunit == null) {
			timeunit = TimeUnit.SECONDS;
		}
		return redisTemplate.getExpire(key, timeunit);
	}

	/**
	 * <p>执行命令获得主题所占用缓存大小</p >
	 *
	 * @param topicName
	 * @return java.lang.String
	 * @since   1.0.0
	 * @author zhk
	 * @date 2024/6/15 9:43 </p >
	 */
	public String redisMemoryUsage(String key) {
		String luaCommand = "return redis.call('MEMORY', 'USAGE', KEYS[1])";
		DefaultRedisScript script = new DefaultRedisScript<>();
		script.setScriptText(luaCommand);
		script.setResultType(Long.class);

		//RedisTemplate redisTemplate = (RedisTemplate)SpringUtil.getBean("redisTemplate");

		return redisTemplate.execute(script, Arrays.asList(key)).toString();
	}

}
