package com.peak.common.util;

import com.peak.prd.base.model.ResponseStatusEnum;
import com.peak.prd.base.model.UploadFile;
import com.peak.prd.base.service.IOssService;
import com.peak.prd.base.util.ExceptionUtil;
import com.peak.prd.config.AppCommonConfig;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.util.UUID;
/**
 * 文件上传
 * @author wzy
 *
 */
public class UploadFileUtil {
	/**
	 * 上传单个文件，不保存lms_files表，存储到永久目录
	 * @param file MultipartFile对象
	 * @param saveRelativePath 文件保存在resource下的子目录，以/结尾，示例：coursepic/，必须传参
	 * @return UploadFile 对象具有属性包含：<br/>
	 * 存储文件表Id(为null)、上传后存储的新文件名、上传文件的原始文件名、上传文件的扩展名、<br/>
	 * 相对resource目录全路径，示例：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.jpg
	 * @throws Exception
	 * @author wzy
	 * @CreateDate 2021年12月4日
	 *
	 */
	public static UploadFile upload(MultipartFile file, String saveRelativePath) throws Exception {
		return upload(file, saveRelativePath, false);
	}

	/**
	 * 上传单个文件，不保存lms_files表，根据uploadTemp判断是否存储到永久目录
	 * @param file MultipartFile对象
	 * @param saveRelativePath 文件保存在resource下的子目录，以/结尾，示例：coursepic/，必须传参
	 * @param uploadTemp 是否上传到临时文件夹，true：上传到临时文件夹，仅保留一天，每天晚上删除前天上传的内容；false:永久保存
	 * @return UploadFile 对象具有属性包含：<br/>
	 * 存储文件表Id(为null)、上传后存储的新文件名、上传文件的原始文件名、上传文件的扩展名、<br/>
	 * 相对resource目录全路径，示例：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.jpg
	 * @throws Exception
	 * @author wzy
	 * @CreateDate 2021年12月4日
	 *
	 */
	public static UploadFile upload(MultipartFile file, String saveRelativePath,boolean uploadTemp) throws Exception {
		return uploadFile(file, saveRelativePath, uploadTemp);
	}

	/**
	 * 上传单个文件，不保存lms_files表，根据uploadTemp判断是否存储到永久目录
	 * @param file MultipartFile对象
	 * @param saveRelativePath 文件保存在resource下的子目录，以/结尾，示例：coursepic/，必须传参
	 * @param uploadTemp 是否上传到临时文件夹，true：上传到临时文件夹，仅保留一天，每天晚上删除前天上传的内容；false:永久保存
	 * @return UploadFile 对象具有属性包含：<br/>
	 * 存储文件表Id(为null)、上传后存储的新文件名、上传文件的原始文件名、上传文件的扩展名、<br/>
	 * 相对resource目录全路径，示例：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.jpg
	 * @throws Exception
	 * @author wzy
	 * @CreateDate 2021年12月4日
	 *
	 */
	private static UploadFile uploadFile(MultipartFile file, String saveRelativePath, boolean uploadTemp) throws Exception {
		if (file == null || saveRelativePath==null) {
			return null;
		}

		//结尾不是/在结尾加上/
		if(saveRelativePath.lastIndexOf("/") != saveRelativePath.length() - 1) {
			saveRelativePath = saveRelativePath + "/";
		}

//		获取原文件名
		String originfilename = file.getOriginalFilename();
		String fileExtention = FileUtil.getFileExtention(originfilename).toLowerCase();
		//检查是否允许上传文件类型
		checkPermission(originfilename,file.getSize());
		//获得保存的全路径，不含文件名
		String fullpath = FileUtil.getSaveFullpath(saveRelativePath);
		String savefilename = FileUtil.getUUIDFilename() + fileExtention;

//		文件存储路径
		String destPath = AppPathUtil.getResourceLocalPath() + fullpath ;
		//临时文件
		if (uploadTemp) {
			destPath = AppPathUtil.getTempLocalPath() + fullpath ;
		}


		File destFilePath = new File(destPath);
		if (!destFilePath.exists()) {
			destFilePath.mkdirs();
		}
//		目的文件
		File destFile = new File(destPath  + savefilename);
//		上传文件
		file.transferTo(destFile);
//		处理OSS上传，临时文件均不上传到oss，类似导入、导出数据的excel文件均不上传到oss上，
		if (!uploadTemp) {
			handleOss(fullpath + savefilename);
		}
//		设置返回对象
		return getUploadFile(originfilename,  savefilename,fileExtention, fullpath + savefilename);

	}



	/**
	 * 通过base64方式上传文件，存储到永久目录
	 * @param base64codefile base64文件字符串
	 * @param saveRelativePath  文件保存在resource下的子目录，以/结尾，示例：coursepic/，必须传参
	 * @return UploadFile 对象具有属性包含：<br/>
	 * 	 存储文件表Id(为null)、上传后存储的新文件名、上传文件的原始文件名、上传文件的扩展名、<br/>
	 * 	 相对resource目录全路径，示例：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.jpg
	 * @author ldc
	 * @CreateDate 2022年2月25日
	 *
	 */
	public static UploadFile upload(String base64codefile, String saveRelativePath) throws Exception {
		if (base64codefile == null || saveRelativePath==null) {
			return null;
		}

		String filename = UUID.randomUUID().toString().replaceAll("-","") + "." + base64codefile.substring(base64codefile.indexOf("/") + 1,
				base64codefile.indexOf(";"));

		return upload(base64codefile, filename, saveRelativePath);
	}



	/**
	 * 通过base64方式上传文件，存储到永久目录
	 * @param base64codefile base64文件字符串
	 * @param filename 文件名称，不包含路径的纯文件名，示例：个人头像.jpg
	 * @param saveRelativePath  文件保存在resource下的子目录，以/结尾，示例：coursepic/，必须传参
	 * @return UploadFile 对象具有属性包含：<br/>
	 * 	 存储文件表Id(为null)、上传后存储的新文件名、上传文件的原始文件名、上传文件的扩展名、<br/>
	 * 	 相对resource目录全路径，示例：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.jpg
	 * @throws Exception
	 * @author wzy
	 * @CreateDate 2021年12月29日
	 *
	 */
	public static UploadFile upload(String base64codefile, String filename, String saveRelativePath) throws Exception {
		return uploadBase64(base64codefile,filename,saveRelativePath,false);
	}

	/**
	 * 通过base64方式上传文件，根据uploadTemp判断是否存储到永久目录
	 * @param base64codefile base64文件字符串
	 * @param saveRelativePath  文件保存在resource下的子目录，以/结尾，示例：coursepic/，必须传参
	 * @param uploadTemp 是否上传到临时文件夹，true：上传到临时文件夹，仅保留一天，每天晚上删除前天上传的内容；false:永久保存
	 * @return UploadFile 对象具有属性包含：<br/>
	 * 	 存储文件表Id(为null)、上传后存储的新文件名、上传文件的原始文件名、上传文件的扩展名、<br/>
	 * 	 相对resource目录全路径，示例：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.jpg
	 * @author ldc
	 * @CreateDate 2022年2月25日
	 *
	 */
	public static UploadFile upload(String base64codefile, String saveRelativePath, boolean uploadTemp) throws Exception {
		if (base64codefile == null || saveRelativePath==null) {
			return null;
		}

		String filename = UUID.randomUUID().toString().replaceAll("-","") + "." + base64codefile.substring(base64codefile.indexOf("/") + 1,
				base64codefile.indexOf(";"));

		return upload(base64codefile, filename, saveRelativePath,uploadTemp);
	}
	/**
	 * 通过base64方式上传文件，根据uploadTemp判断是否存储到永久目录
	 * @param base64codefile base64文件字符串
	 * @param filename 文件名称，不包含路径的纯文件名，示例：个人头像.jpg
	 * @param saveRelativePath  文件保存在resource下的子目录，以/结尾，示例：coursepic/，必须传参
	 * @param uploadTemp 是否上传到临时文件夹，true：上传到临时文件夹，仅保留一天，每天晚上删除前天上传的内容；false:永久保存
	 * @return UploadFile 对象具有属性包含：<br/>
	 * 	 存储文件表Id(为null)、上传后存储的新文件名、上传文件的原始文件名、上传文件的扩展名、<br/>
	 * 	 相对resource目录全路径，示例：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.jpg
	 * @throws Exception
	 * @author wzy
	 * @CreateDate 2021年12月29日
	 *
	 */
	public static UploadFile upload(String base64codefile, String filename, String saveRelativePath, boolean uploadTemp) throws Exception {
		return uploadBase64(base64codefile,filename,saveRelativePath,uploadTemp);
	}

	/**
	 * 通过base64方式上传文件
	 * @param base64codefile base64文件字符串
	 * @param filename 文件名称，不包含路径的纯文件名，示例：个人头像.jpg
	 * @param saveRelativePath  文件保存在resource下的子目录，以/结尾，示例：coursepic/，必须传参
	 * @param uploadTemp 是否上传到临时文件夹，true：上传到临时文件夹，仅保留一天，每天晚上删除前天上传的内容；false:永久保存
	 * @return UploadFile 对象具有属性包含：<br/>
	 * 	 存储文件表Id(为null)、上传后存储的新文件名、上传文件的原始文件名、上传文件的扩展名、<br/>
	 * 	 相对resource目录全路径，示例：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.jpg
	 * @throws Exception
	 * @author wzy
	 * @CreateDate 2021年12月29日
	 *
	 */
	private static UploadFile uploadBase64(String base64codefile, String filename, String saveRelativePath, boolean uploadTemp) throws Exception {
		if (base64codefile == null || filename == null || saveRelativePath==null) {
			return null;
		}


		//结尾不是/在结尾加上/
		if(saveRelativePath.lastIndexOf("/") != saveRelativePath.length() - 1) {
			saveRelativePath = saveRelativePath + "/";
		}

//		获取文件扩展名
		String fileExtention = FileUtil.getFileExtention(filename).toLowerCase();


		//base64上传只支持图片
		if(!SecurityUtil.isAllowImgext(filename)) {
			ExceptionUtil.handle(ResponseStatusEnum.FILE_IMG_DOWNLOAD_FAILD);
		} else if(!Base64Util.base64IsImage(base64codefile)) {
			//base64字符串不是图片格式
			ExceptionUtil.handle(ResponseStatusEnum.FILE_IMG_DOWNLOAD_FAILD);
		}
		//获得保存的全路径，不含文件名
		String fullpath = FileUtil.getSaveFullpath(saveRelativePath);
		String savefilename = FileUtil.getUUIDFilename() + fileExtention;
//		文件存储路径
		String destPath = AppPathUtil.getResourceLocalPath() + fullpath ;
		//临时文件
		if (uploadTemp) {
			destPath = AppPathUtil.getTempLocalPath() + fullpath ;
		}

		File destFilePath = new File(destPath);
		if (!destFilePath.exists()) {
			destFilePath.mkdirs();
		}

//		上传base64字符串中包含data:image/jpeg;base64,需要截取掉
		if(base64codefile.indexOf(",") > -1) {
			base64codefile = base64codefile.substring(base64codefile.indexOf(",") + 1);
		}

		// Base64解码
		byte[] filebytes = Base64Util.decode(base64codefile);

		//图片大小
		long imagesize = filebytes.length;
		checkPermission(filename,imagesize);

//		下面循环因为网上都这么写，防止报错也保留了，具体是否起作用未知
//		测试了几张图片，要不要for循环都没问题
		for (int i = 0; i <filebytes.length; ++i) {
			if (filebytes[i] < 0) {// 调整异常数据
				filebytes[i] += 256;
			}
		}
		// 文件写入
		OutputStream out = new FileOutputStream(destPath  + savefilename);
		out.write(filebytes);
		out.flush();
		out.close();

//		处理OSS上传，临时文件均不上传到oss，类似导入、导出数据的excel文件均不上传到oss上，
		if (!uploadTemp) {
			handleOss(fullpath + savefilename);
		}

//		设置返回对象
		return getUploadFile(filename, savefilename, fileExtention, fullpath + savefilename);

	}


	/**
	 * 上传单个文件，如果使用oss，会上传到oss
	 * @param srcfile 源文件对象
	 * @param saveRelativePath 文件保存在resource下的子目录，以/结尾，示例：coursepic/，必须传参
	 * @return UploadFile 对象具有属性包含：<br/>
	 * 	 存储文件表Id、上传后存储的新文件名、上传文件的原始文件名、上传文件的扩展名、<br/>
	 * 	 相对resource目录全路径，示例：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.jpg
	 * @author wzy
	 * @date 2023/7/30 9:47
	 */
	public static UploadFile upload(File srcfile, String saveRelativePath, boolean uploadTemp) throws Exception {
		if (srcfile == null || saveRelativePath==null) {
			return null;
		}

		//结尾不是/在结尾加上/
		if(saveRelativePath.lastIndexOf("/") != saveRelativePath.length() - 1) {
			saveRelativePath = saveRelativePath + "/";
		}

//		获取原文件名
		String originfilename = srcfile.getName();
		String fileExtention = FileUtil.getFileExtention(originfilename).toLowerCase();
		//检查是否允许上传文件类型
		checkPermission(originfilename,srcfile.length());
		//获得保存的全路径，不含文件名
		String fullpath = FileUtil.getSaveFullpath(saveRelativePath);
		String savefilename = FileUtil.getUUIDFilename() + fileExtention;

//		文件存储路径
		String destPath = AppPathUtil.getResourceLocalPath() + fullpath ;
		//临时文件
		if (uploadTemp) {
			destPath = AppPathUtil.getTempLocalPath() + fullpath ;
		}


		File destFilePath = new File(destPath);
		if (!destFilePath.exists()) {
			destFilePath.mkdirs();
		}
//		目的文件
		File destFile = new File(destPath  + savefilename);
//		上传文件
		FileUtil.copyFile(srcfile,destFile);
//		处理OSS上传，临时文件均不上传到oss，类似导入、导出数据的excel文件均不上传到oss上，
		if (!uploadTemp) {
			handleOss(fullpath + savefilename);
		}
//		设置返回对象
		return getUploadFile(originfilename,  savefilename,fileExtention, fullpath + savefilename);
	}


	/**
	 * 检测是否允许上传文件及是否满足允许的上传文件尺寸
	 * @param filename 文件扩展名
	 * @param filesize 文件大小，单位：字节
	 * @author wzy
	 * @date 2023/7/30 11:02
	 */
	private static void checkPermission(String filename,Long filesize) {
		SecurityUtil.checkUploadFilePermission(filename,filesize);
	}

	/**
	 * 采用oss上传
	 * @param fullpathfilename 相对resource目录全路径文件全路径,示例：coursepic/2021/08/20/ae37d4bd30954edb97985db52d85da34.jpg
	 * @author wzy
	 * @date 2023/7/30 10:26
	 */
	private static void handleOss(String fullpathfilename) {
		AppCommonConfig config = SpringUtil.getBean(AppCommonConfig.class);
		//使用oss存储
		if (config.getOssEnable()) {
			//上传文件到OSS
			IOssService ossService = (IOssService) SpringUtil.getBean(config.getOssServiceName());
			ossService.uploadFile(AppPathUtil.getResourceLocalPath(), fullpathfilename);
		}
	}
	/**
	 * 获得UploadFile对象
	 * @param srcfilename 源文件名
	 * @param destfilename 目的文件名
	 * @param fileExtention 文件扩展名
	 * @param fullpath 相对resource目录全路径，示例：coursepic/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.jpg
	 * @return com.peak.prd.base.model.UploadFile
	 * @author wzy
	 * @date 2023/7/30 10:37
	 */
	private static UploadFile getUploadFile(String srcfilename, String destfilename,String fileExtention, String fullpath) {
		UploadFile uploadFile = new UploadFile();
		uploadFile.setSavefilename(destfilename);
		uploadFile.setSrcfilename(srcfilename);
		uploadFile.setFileNameExt(fileExtention);
		uploadFile.setFullpath(fullpath);
		return uploadFile;
	}


}
