package com.peak.framework.common.dict;
/**
 * 功能描述：临时简单的数据字典 以后可以扩展成更高级的
 */

import com.peak.framework.common.cache.ICache.CacheType;
import com.peak.framework.common.cache.SystemCache;
import com.peak.framework.common.util.PropertiesHelper;
import com.peak.framework.util.SpringContainer;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
/**
 * 功能描述：配置文件支持的字典类
 *  1.字典key不区分大小写 统一采用大写形式
 *  2.字典分为:静态字典 动态字典
 *   2.1 目前静态字典通过config/dict/dict.properites配置文件配置，在系统启动的时候加载一次
 *         配置格式为： dictCode=key1:name1;key2=name2;key3=name3
 *   2.2 动态字典为在 com.peak.framework.dict.dbdict.model.mapper.DatabaseDictMapper.xml配置的sql字典
 *        可以在类路径下任意建立 mybatis配置文件 
 *        只要保证 namespace="DatabaseDict",在系统启动时候被mybatis加载。
 *        动态字典文件分为两类 
 *        1. 以DB_命名开始的 
 *           id 格式 ：id="DatabaseDict.DB_xxxxxx"  
 *           视图格式:select NAME,VALUE,xxx,yyyy FROM [tableName]
 *           举例 
 *           	<select id="DatabaseDict.DB_MENU_TYPE" resultType="java.util.HashMap">
 *			    	SELECT  M_NAME  NAME,M_TYPE VALUE FROM REDFIRE_MENU_TYPE  
 *			    </select>
 *           这类字典没有传入参数，通常是加载特定字典集合下的所有数据，缓存在内存中使用。
 *        2. 以DBP_命名开始的
 *             id 格式 ：id="DatabaseDict.DBP_xxxxxx"  
 *             视图格式:select NAME,VALUE,xxx,yyyy FROM [tableName] where key=[code]
 *             举例 
 *           <select id="DatabaseDict.DBP_LESSONBYID" resultType="java.util.HashMap">
 *		    	SELECT LE_ID VALUE,LE_NAME NAME 
 *		    	FROM LESSON  WHERE LE_ID=#{key , jdbcType=DECIMAL}
 *		    </select> 
 *          这类字典用在当表中有超量的数据，当前场景下只会使用部分数据。将使用 ‘最少使用'算法缓存在
 *          系统缓存中供翻译字典使用，不需要拿到字典全集，例如：有10万机构数据，机构Id 到机构名称的翻译。    
 *  3.动态字典的生命周期： 加载，缓存，修改，清空，
 *  4.config/dict/dictClear.properties 配置了那些类的   对数据库的更改会引起动态字典的缓存和数据库中的
 *    数据不一致，
 *  5.clearDict 方法清空与特定类相关的所有字典缓存,
 *  6.动态数据字典的加载方式是:如果用户强制reload，则一定重新加载。如果用户没有强制reload,如果缓存
 *    中没有数据才会依据DatabaseDictMapper.xml 加载数据，并更新缓存。
 * @author zhaoguodong
 */
public class DictUtil {
	/**
	 * 记录类名和相关联的动态数据库字典集合
	 */
	private static  Map<String,String[]> CLASS_DB_DICT=new HashMap<String,String[]>();
	public static String ITEM_LIST_P="#ITEM_LIST";//字典list
	private static String DBDICT_CACHE_NAME="DB_DICT";//数据库字典
	private static String DBPDICT_CACHE_NAME="DBP_DICT";//数据库字典
	private static String DICT_CACHE_NAME="DICT";//properties字典
    static {
			 // 静态字典
    	  //静态字典 采用内存map缓存
	      SystemCache.create(DICT_CACHE_NAME,CacheType.MapCache,200 );
	      Map<String,Object>   dictMap=PropertiesDict.reloadProperties(ITEM_LIST_P);
	      
	      SystemCache.set(DICT_CACHE_NAME,dictMap);
		  //动态字典 使用ehcache缓存
		  SystemCache.create(DBDICT_CACHE_NAME,CacheType.EhCache,0);
		  //动态字典 使用ehcache缓存 只做翻译用的字典
		  SystemCache.create(DBPDICT_CACHE_NAME,CacheType.EhCache,0);
		 
		  try {
			 //数据库动态字典  这个配置文件配置了一些dao 当进行, 数据库数据改动时候，应该清空的字典缓存
			 Properties dictClearConfig =new Properties();
			 PropertiesHelper.loadAllPropertiesFromClassLoader(dictClearConfig,"config/dict/dictClear.properties");
			 initClearMap(dictClearConfig);//dbz字典的清空机制
		   } catch (IOException e) {
			 e.printStackTrace();
		   }
	 }
     public static void reloadProperties(){
    	   Map<String,Object>   dictMap=PropertiesDict.reloadProperties(ITEM_LIST_P);
	       SystemCache.set(DICT_CACHE_NAME,dictMap);
     }
	 /**
	  * 功能描述：初始化dao的和字典的对应关系
	  * @param dictConfig
	  */
	 private static void initClearMap(Properties dictConfig){
		for(Object key :dictConfig.keySet()){
		    String data=(String) dictConfig.get(key);
		    if(data==null||"".equals(data.trim())) continue;
		    String[] dictCodes =data.split(";");
		    CLASS_DB_DICT.put(key.toString(),dictCodes);
	    }
	 }
	 /**
	  * 功能描述：清空当前类想对应的数据字典
	  * @param clazz
	  */
	public static void clearDict(Class clazz){
		 String[] dictCodes=CLASS_DB_DICT.get(clazz.getName());
		 if(dictCodes==null) return;
		 for(int i=0;i<dictCodes.length;i++){
			    String dictCode=dictCodes[i].toUpperCase();
			    if(dictCode.startsWith("DB_")){
				     SystemCache.delete(DictUtil.DBDICT_CACHE_NAME,dictCode+ITEM_LIST_P);
				    SystemCache.delete(DictUtil.DBDICT_CACHE_NAME,dictCode);
			    }
			    if(dictCode.startsWith("DBP_")){
			    	 SystemCache.delete(DBPDICT_CACHE_NAME,dictCodes[i].toUpperCase());
				}	 
		 }
	}
	/**
	 * 功能描述：加载数据库字典(动态字典）
	 * @param dictCode
	 * @return
	 */
	 public static  void  loadDBDict(String dictCode){
  	      if(dictCode==null)return ;
  	      dictCode= dictCode.toUpperCase();
  	      if(!dictCode.startsWith("DB_")) return ;
  	      
  	      IDictService dictService= (IDictService) SpringContainer.getObject("dictService");
  	      List<Map> list=dictService.loadDictList(dictCode);
  	      List<Item> itemList=new LinkedList<Item>();
  	      Map<String,String> dictMap=new HashMap<String,String>();
  	      if(list==null) return ;
  	      for(Map map:list){
  	    	Item item=new Item(map);
  	    	itemList.add(item);
  	    	dictMap.put(item.value, item.name);
  	      }
  	      SystemCache.set(DictUtil.DBDICT_CACHE_NAME,dictCode+ITEM_LIST_P, itemList);
  	      SystemCache.set(DictUtil.DBDICT_CACHE_NAME,dictCode, dictMap);
     }
	 public static void reloadDBDictName(String dictCode,String... keys){
		  if(dictCode==null)return ;
  	      dictCode= dictCode.toUpperCase();
  	      if(dictCode.startsWith("DB_")){
  	    	    loadDBDict(dictCode);
  	      }
  	      if(dictCode.startsWith("DBP_")){
  	    	 for(String key:keys){
  	    		loadDBPDict(dictCode,key); 
  	         }
	      }
	 }
	 /**
	  * 功能描述：加载翻译字典
	  * @param dictCode
	  * @param key
	  * @return
	  */
	 public static  String  loadDBPDict(String dictCode,String key){
	     if(dictCode==null)return "";
 	         dictCode= dictCode.toUpperCase();
 	      if(!dictCode.startsWith("DBP_")) return "";
 	        IDictService dictService= (IDictService) SpringContainer.getObject("dictService");
 	        Map dictItem=dictService.loadDict(dictCode,key);
 	        if(dictItem==null) return "";
  	        Map<String,String> dictMap=(Map<String, String>) SystemCache.get(DictUtil.DBPDICT_CACHE_NAME,dictCode);
 	        if(dictMap==null){
 	        	dictMap=new HashMap<String,String> ();
 	        }
	    	Item item=new Item(dictItem);
	    	dictMap.put(item.value, item.name);
 	        SystemCache.set(DictUtil.DBPDICT_CACHE_NAME,dictCode, dictMap);
 	        return item.name;
    }
	 /**
	  * 加载特定域 没有缓存
	  * @param dictCode
	  * @param domainId
	  * @return
	  */
	public static  List<Item> loadDBDict(String dictCode,Long domainId){
 	       if(dictCode==null) return Collections.emptyList();
 	       dictCode= dictCode.toUpperCase();
 	       if(!dictCode.startsWith("DB_")) Collections.emptyList() ;
 	       Map params=new HashMap();
 	       params.put("domainId", domainId);
 	       IDictService dictService= (IDictService) SpringContainer.getObject("dictService");
 	       List<Map> list=dictService.loadDictList(dictCode,params);
 	       List<Item> itemList=new LinkedList<Item>();
  	       if(list==null) Collections.emptyList();
  	       for(Map map:list){
  	    	 Item item=new Item(map);
  	    	 itemList.add(item);
  	       }
  	       return itemList;
  	      
    }
	//-------------------------------------------------------------------------------------
	 /**
	  * 功能描述：加载用户信息
	  * @param dictCode
	  */
	
	 public static  void  removeDBDict(String dictCode){
  	     SystemCache.delete(DictUtil.DBDICT_CACHE_NAME, dictCode.toUpperCase()+ITEM_LIST_P);
  	     SystemCache.delete(DictUtil.DBDICT_CACHE_NAME, dictCode.toUpperCase());
  	     return ;
     }
	 /**
	  * 功能描述：判断字典是否存在
	  * @param dictCode
	  * @return
	  */
	 public static boolean exists(String dictCode){
		    dictCode=dictCode.toUpperCase();
		    boolean dbDict=dictCode.startsWith("DB_");
		    Object dictMap=null;
		    if(dbDict) {
		    	dictMap= SystemCache.get(DictUtil.DBDICT_CACHE_NAME,dictCode);
	        }else{
	        	dictMap = getDictCacheMap(dictCode);
	        }
	        if(dictMap!=null) 
	    	  return true;
	       return false;
	 }
	private static Object getDictCacheMap(String dictCode) {
			 if (dictCode == null)  return null;
			 java.util.Locale locale=I18N.LOCALETHREADLOCAL.get();
			 Object dictMap=null;
	         if(locale!=null && !(locale.getLanguage().indexOf("zh")>-1)) {
		         dictMap= SystemCache.get(DictUtil.DICT_CACHE_NAME,"EN_"+dictCode);
	         }
	         if(dictMap==null){
	        	 dictMap= SystemCache.get(DictUtil.DICT_CACHE_NAME,dictCode);
	         }	
		     return dictMap;
	}
	 /**
	  * 功能描述：返回 配置文件 和 db中 的 字典 
	  * @param code
	  * @return
	  */
     public static Map<String,String> getDict(String dictCode){
    	       if(dictCode==null ) return null;
    	       dictCode=dictCode.toUpperCase();
    	       boolean dbDict=dictCode.startsWith("DB_");
    	       if(dbDict) {
    	    	    Map<String, String> returnValue = (Map<String, String>) SystemCache.get(DictUtil.DBDICT_CACHE_NAME,dictCode);
    	    	    if(returnValue==null||returnValue.size()==0){
    	    	    	 DictUtil.loadDBDict(dictCode);
    	    	    	 returnValue = (Map<String, String>) SystemCache.get(DictUtil.DBDICT_CACHE_NAME,dictCode);
    	    	    }
    	    	    return returnValue;
		        }else{
		        	return (Map<String, String>)getDictCacheMap(dictCode);
		        }
     }
     public static List<Item> getSortDict(String dictCode){
		        if(dictCode==null ) return null;
		        boolean dbDict=dictCode.startsWith("DB_");
		        dictCode=dictCode.toUpperCase();
		        if(dbDict) {
		        	    List<Item> returnValue = (List<Item>) SystemCache.get(DictUtil.DBDICT_CACHE_NAME,dictCode+ITEM_LIST_P);
	    	    	    if(returnValue==null){
	    	    	    	 DictUtil.loadDBDict(dictCode);
	    	    	    	 returnValue = (List<Item>) SystemCache.get(DictUtil.DBDICT_CACHE_NAME,dictCode+ITEM_LIST_P);
	    	    	    }
	    	    	    return returnValue;
		        }else{
		        	    return (List<Item>) getDictCacheMap(dictCode+ITEM_LIST_P);
		        }
     }
     /**
      * 功能描述：获取特定字典 特定值的名称
      * @param dictCode
      * @param value 
      * @return
      */
     public static String getDictName(String dictCode,String value){
    	   if(dictCode==null ) return null;
	       dictCode=dictCode.toUpperCase();
	       // 以DBP_开始的字典需要的key参数 只能用来翻译
	       boolean dbpDict=dictCode.startsWith("DBP_");
	       if(dbpDict) {
	    	    Map<String, String> dictMap = (Map<String, String>) SystemCache.get(DictUtil.DBPDICT_CACHE_NAME,dictCode);
	    	    if(dictMap==null||dictMap.size()==0){
	    	    	 return DictUtil.loadDBPDict(dictCode,value);
	    	    }
	    	    String name= dictMap.get(value);
	    	    if(name==null||"".equals(name)){
	    	    	name=DictUtil.loadDBPDict(dictCode,value);
	    	    }
   	    	    return name;
	       }
    	   Map<String,String> dict=DictUtil.getDict(dictCode);
    	   if(dict!=null){
    		 String returnValue= dict.get(value);
    		 if(returnValue==null) return "";
    		 return returnValue;
    	  }
    	  return "";
     }
     public static void main(String args[]){
    	 Map<String,String> dict=getDict("ZdrAttributes.valueType");
    	  for(String key:dict.keySet()){
//    		  System.out.println(key +":"+dict.get(key));
    	  }
    	 // System.out.println(PageUtil.getDictOption("ZdrAttributes.valueType"));
     }
     /**
      * 
      * 功能描述：
      * @author zhao
      * @version 2011-8-12 v1.0
      */
     public static class Item implements java.io.Serializable{
     	/**
     	 * 每一项数据
     	 */
          private Map data;
          private String name;
		  private String value;
     	  public Item(String value,String name){
         	 this.value=value;
         	 this.name=name;
          }
     	  public Map getEntity(){
     		  return data;
     	  }
          public Item(Map dataMap){
         	 Object nameObj=dataMap.get("NAME");
     	     Object valueObj=dataMap.get("VALUE");
     	     if(valueObj!=null)
     	    	 this.value=valueObj.toString();
     	     if(nameObj!=null)
     	    	 this.name=nameObj.toString();
         	 this.data=dataMap;
          }
          public String getName() {
  			return name;
	      }
	 
	  	  public String getValue() {
	  			return value;
	  	  }	
     }
}

