package com.peak.framework.key;

import org.apache.commons.logging.Log;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * 
 *  tweeter的snowflake 移植到Java:
 *  (a) id构成: 41位的时间前缀 + 5位datacenter+5位的机器id + 12位的sequence避免并发的数字(12位不够用时强制得到新的时间前缀);
 *  (b) 对系统时间的依赖性非常强，需关闭ntp的时间同步功能。当检测到ntp时间调整后，将会拒绝分配id
 *  (c) 每毫秒最多可产生4095个主键 
 *  workerId  [0,31]
 *  datacenterId  [0,31]
 */
public class SnowflakeId {
		private static Log LOGGER = org.apache.commons.logging.LogFactory.getLog(SnowflakeId.class);
		private static final long twepoch = 1451606400000L;//2016.1.1 8:00:00
		private final long workerIdBits = 5L;
		private final long datacenterIdBits = 5L;
		//31
		private final long maxWorkerId = -1L ^ (-1L << workerIdBits);
		private final long maxDatacenterId = -1L ^ (-1L << datacenterIdBits);
		private final long sequenceBits = 12L;
		private final long workerIdShift = sequenceBits;
		private final long datacenterIdShift = sequenceBits + workerIdBits;
		private final long timestampLeftShift = sequenceBits + workerIdBits + datacenterIdBits;
		//4095
		private final long sequenceMask = -1L ^ (-1L << sequenceBits);
		private long lastTimestamp = -1L;
		
		private long workerId = 0;
		private long datacenterId = 0;
		private long sequence = 0L;

		/**
		 * 
		 * @param workerId  [0,31]
		 * @param datacenterId  [0,31]
		 */
		public SnowflakeId(long datacenterId,long workerId) {
			checkParams(workerId,datacenterId);
			this.datacenterId = datacenterId;
			this.workerId = workerId;
			
		}
		/**
		 * 
		 * @return
		 */
		public synchronized long next() {

			long id = 0L;

			long timestamp = System.currentTimeMillis();
			
			checkUpdatTime(timestamp);
			
			if (timestamp == lastTimestamp) {
				 sequence = (1 + sequence) & sequenceMask;
				 if(0 == sequence) { //这一毫秒获取的id数超过>4095
					 //等待下一个毫秒的到来, 保证返回的毫秒数在参数lastTimestamp之后
					 timestamp = System.currentTimeMillis();
					 while(timestamp <= lastTimestamp) {
						timestamp = System.currentTimeMillis();
					 }
				 }
			} else {
				 sequence = 0;
			}
			
			lastTimestamp = timestamp;
			
			id = ((timestamp - twepoch) << timestampLeftShift) | 
					(datacenterId << datacenterIdShift) | 
					(workerId << workerIdShift) | 
					sequence;

			return id;
		}
		/**
		 * 每次最少跳过  4095+(4095-sequence);
		 * 方法最少运行2毫秒，
		 * @param amount
		 * @return
		 */
		public synchronized long jump(int amount) {
			// if amount<4095-sequence   then lastTimestamp+1;
			// else lastTimestamp+(  (amount-(4095-sequence))/amount+1  ) 
			//直接写为 amount/sequenceMask+2
			    long num=amount/sequenceMask+2;  
			    long seq=next();
			    long timestamp = lastTimestamp;
			    while(timestamp < lastTimestamp+num) {
				    timestamp = System.currentTimeMillis();
			    }
			    lastTimestamp=System.currentTimeMillis();
			    sequence = 0;
			    return seq;
		}
		private void checkUpdatTime(long timestamp) {
			if (timestamp < lastTimestamp) {
				 datacenterId++;
				 long yel=(lastTimestamp-timestamp)/1000/60;
				 LOGGER.warn("系统时间被回调了  ，为保证不发生主键冲突 ， 请不要在 （"+yel +"）  分钟内不要重启服务器 ,如在（"+yel +"）  分钟内重启服务器，请配置  pk_databaseId=1 （applicaiton.properties");
			     //throw new RuntimeException(String.format("clock moved backwards. refusing to generate id for %d milliseconds.", lastTimestamp - timestamp));
			}
		}
		private void checkParams(long workerId, long datacenterId){
			if (workerId > maxWorkerId || workerId < 0) {
				throw new IllegalArgumentException(String.format("workerId can't be greater than %d or less than 0.", maxWorkerId));
			}
			if (datacenterId > maxDatacenterId || datacenterId < 0) {
				throw new IllegalArgumentException(String.format("datacenterId can't be greater than %d or less than 0.", maxDatacenterId));
			}
		}
		
		public static String parse(long  seq){
			//5位datacenterId+5位的机器id + 12位的sequence避免并发的数字(12位不够用时强制得到新的时间前缀);
			long time=seq>>22;//前42位 时间 time 右移动22
			Date d=new Date();
			d.setTime(twepoch+time);
			DateFormat dateFormate=new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
			String timeStr=dateFormate.format(d);
			
			  //seq & (0-42|1-5|0-5|0-12) seq&0x3e0000>>17
			  long datacenterId=(seq<<42>>59);
			  //long datacenterId=(seq&0x3e0000)>>17;
			  //seq & (0-42|0-5|1-5|0-12) seq&0x1F000>>12
			  long workerId=(seq<<47>>59);		
			  //long workerId=(seq&0x1F000)>>12;
			  //seq & (0-42|0-5|0-5|1-12) seq&0xFFF
			  long seqS=(seq<<52>>52);
			 //long seqS=(seq&0xFFF);
			 
			String v= timeStr+"|"+datacenterId+"|"+workerId+"|"+seqS;
			return v;
		}
		public static void main(String[] args){
			 SnowflakeId sf=new SnowflakeId(17,9);
		     long timestamp = System.currentTimeMillis();
			 long id = 0l;
			 for(int i=0;i<100000;i++){
				 id = sf.next();

			 }


		   
		}
		
	}