package com.peak.prd.base.advise;

import com.peak.common.util.AOPUtil;
import com.peak.common.util.SpringUtil;
import com.peak.common.util.StringUtil;
import com.peak.prd.base.annotation.ClearCache;
import com.peak.prd.base.service.imple.PrdBaseService;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.After;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.stereotype.Component;

/**
 * @description @ClearCache注解拦截器
 * @author ZhengJia
 * @version 1.0
 * @date 2023/11/8 2:29 PM
 */
@Aspect
@Component
public class ClearCacheAdvise {

    // 使用Spring提供的工具获取方法参数名
    @After("@annotation(clearCache)")
    public void after(JoinPoint joinPoint, ClearCache clearCache) throws Exception {

        //获得Spel上下文
        StandardEvaluationContext standardEvaluationContext = AOPUtil.getContextVariables(joinPoint);

        // 按id清除缓存
        this.clearCacheById(joinPoint, clearCache, standardEvaluationContext);

        // 多参数清除缓存
        this.clearCacheByValues(joinPoint, clearCache, standardEvaluationContext);
    }

    /**
     * @description 按id清除缓存
 	 * @param joinPoint
 	 * @param clearCache
 	 * @param standardEvaluationContext
     * @return
     * @author ZhengJia
     * @date 2025/1/13 17:11
     **/
    private void clearCacheById(JoinPoint joinPoint, ClearCache clearCache, StandardEvaluationContext standardEvaluationContext) throws Exception {
        //获得注解参数value
        String value = clearCache.value();
        //获得keys值
        String[] keys = clearCache.keys();
        List<String> keyList = new ArrayList<>(Arrays.asList(keys));;
        //将value统一转换到keys中进行统一处理
        if(!StringUtil.isNullorEmptyStr(value)) {
            keyList.add(value);
        }
        //多个value时，循环清除缓存
        for (String key : keyList) {
            Long id = AOPUtil.getSpelValue(standardEvaluationContext, key, Long.class);
            if (id == null) {    // id为null不执行清除缓存操作
                continue;
            }

            // 注解所在类
            Class clazz = joinPoint.getTarget().getClass();
            Object service = SpringUtil.getBean(clazz);
            if (service instanceof PrdBaseService) { // 如果是生成的PlusService或者CustService
                ((PrdBaseService) service).clearCache(id);
            } else {    // 用户自定义Service，如果实现了clearCache方法也可以
                Method clearCacheMethod = null;
                try {
                    clearCacheMethod = clazz.getDeclaredMethod("clearCache", Long.class);
                } catch (Exception e) {
                    e.printStackTrace();
                }
                if (clearCacheMethod == null) {
                    throw new RuntimeException("ClearCache注解所在类没有实现clearCache方法");
                }
                clearCacheMethod.invoke(service, id);
            }
        }
    }

    /**
     * @description 多参数清除缓存
 	 * @param joinPoint
 	 * @param clearCache
 	 * @param standardEvaluationContext
     * @return
     * @author ZhengJia
     * @date 2025/1/13 17:12
     **/
    private void clearCacheByValues(JoinPoint joinPoint, ClearCache clearCache, StandardEvaluationContext standardEvaluationContext) throws Exception {
        // 获取多参数
        String[] values = clearCache.values();
        if(values == null || values.length == 0) {
            return;
        }

        List<String> paramList = new ArrayList<>();
        for (String value : values) {
            String valueStr = AOPUtil.getSpelValue(standardEvaluationContext, value, String.class);
            paramList.add(valueStr);
        }

        // 注解所在类
        Class clazz = joinPoint.getTarget().getClass();
        Object service = SpringUtil.getBean(clazz);
        if (service instanceof PrdBaseService) { // 如果是生成的PlusService或者CustService
            ((PrdBaseService) service).clearCache(paramList);
        } else {    // 用户自定义Service，如果实现了clearCache方法也可以
            Method clearCacheMethod = null;
            try {
                clearCacheMethod = clazz.getDeclaredMethod("clearCache", List.class);
            } catch (Exception e) {
                e.printStackTrace();
            }
            if (clearCacheMethod == null) {
                throw new RuntimeException("ClearCache注解所在类没有实现clearCache方法");
            }
            clearCacheMethod.invoke(service, paramList);
        }
    }

}
