package com.peak.prd.base.advise;

import com.peak.prd.base.model.ApiResult;
import com.peak.prd.base.model.IRespStatusEnum;
import com.peak.prd.exception.DataAuthException;
import com.peak.prd.exception.PrdBaseException;
import com.peak.prd.exception.PrepareException;
import com.peak.prd.exception.SessionNullException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

// https://www.cnblogs.com/magicalSam/p/7198420.html

@ControllerAdvice
@Slf4j
public class PrdExceptionHandler {
//	/**
//     * 应用到所有@RequestMapping注解方法，在其执行之前初始化数据绑定器
//     * @param binder
//     */
//    @InitBinder
//    public void initBinder(WebDataBinder binder) {}
//
//    /**
//     * 把值绑定到Model中，使全局@RequestMapping可以获取到该值
//     * @param model
//     */
//    @ModelAttribute
//    public void addAttributes(Model model) {
//        model.addAttribute("author", "Magical Sam");
//    }

	/**
	 * 产品定义各种异常拦截处理，通常为业务自己判断的逻辑报错
	 */
	@ResponseStatus(value = HttpStatus.OK)
	@ResponseBody
	@ExceptionHandler(PrdBaseException.class)
	public ApiResult prdExceptionHandler(PrdBaseException ex) {
		ApiResult ar = new ApiResult();
		IRespStatusEnum responseStatusEnum = ex.getResponseStatusEnum();
		//通过ExceptionUtil.handle(String code, String msg)方法调用
		if (responseStatusEnum == null) {
			ar.setStatus(ex.getStatus());
			ar.setCode(ex.getCode());
			ar.setMsg(ex.getMsg());
		} else {
			//通过枚举传递的值
			ar.setStatus(responseStatusEnum.getStatus());
			ar.setCode(responseStatusEnum.getCode());
			ar.setMsg(responseStatusEnum.getMsg());
		}
		return ar;
	}



	/**
	 * 请求参数检查异常，
	 * https://blog.csdn.net/weixin_42206151/article/details/134562855
	 */
	@ResponseStatus(value = HttpStatus.OK)
	@ResponseBody
	@ExceptionHandler(value = MethodArgumentNotValidException.class)
	public ApiResult dataValidExceptionHandler(MethodArgumentNotValidException ex) {
		ex.printStackTrace();
		BindingResult br = ex.getBindingResult();
		ApiResult ar = new ApiResult();
		ar.setCode("a_input_parameter_error");
		ar.setMsg("入参错误[a]：" + this.getMsg(br));
		ar.setStatus(ApiResult.STATUS_CLIENT_ERR);
		ar.setData(getErrors(br));
		return ar;
	}

	/**
	 * 请求参数检查异常
	 */
	@ResponseStatus(value = HttpStatus.OK)
	@ResponseBody
	@ExceptionHandler(value = ConstraintViolationException.class)
	public ApiResult dataValidExceptionHandler(ConstraintViolationException e) {
		e.printStackTrace();
		ApiResult ar = new ApiResult();
		ar.setCode("b_input_parameter_error");
		ar.setMsg("入参错误[b]：" + e.getMessage());
		ar.setStatus(ApiResult.STATUS_CLIENT_ERR);
		return ar;
	}

	/** 未登录异常 */
	@ResponseStatus(value = HttpStatus.OK)
	@ResponseBody
	@ExceptionHandler(value = SessionNullException.class)
	public ApiResult<?> sessionExceptionHandler(SessionNullException ex) {

		ApiResult ar = new ApiResult<>();
		ar.setStatus(401); // 401表示未认证
		ar.setCode("not login");
		ar.setMsg("用户未登录[User not logged in]");

		return ar;
	}

	/** 数据鉴权异常 */
	@ResponseStatus(value = HttpStatus.OK)
	@ResponseBody
	@ExceptionHandler(value = DataAuthException.class)
	public ApiResult<?> dataAuthExceptionHandler(DataAuthException ex) {
		log.warn(ex.getMessage(), ex);

		ApiResult ar = new ApiResult<>();
		ar.setStatus(403); // 403表示已经认证，但权限不足
		ar.setCode("Data beyond authority");
		ar.setMsg("数据访问越权[Data beyond authority]");

		return ar;
	}

	/** 业务逻辑执行前，数据准备异常 */
	@ResponseStatus(value = HttpStatus.OK)
	@ResponseBody
	@ExceptionHandler(value = PrepareException.class)
	public ApiResult<?> prepareExceptionHandler(PrepareException ex) {

		ApiResult ar = new ApiResult<>();
		ar.setStatus(400);
		ar.setCode("PrepareError");
		ar.setMsg("PrepareError");

		return ar;
	}



//	/**
//     * 文件上传尺寸太大
//     * https://bbs.csdn.net/topics/394361680?list=67063123
//     */
//	@ResponseStatus(value = HttpStatus.BAD_REQUEST)
//    @ResponseBody
//    @ExceptionHandler(value = MaxUploadSizeExceededException.class)
//    public ApiResult fileSizeExceptionHandler(MaxUploadSizeExceededException ex) {
//        ApiResult ar = new ApiResult();
//		ar.setStatus(ApiResult.STATUS_CLIENT_ERR);
//		ar.setCode("upload_file_size_too_big");
//		ar.setMsg("上传文件太大");
//		
//		return ar;
//    }
	/**
	 * 获取BO中的校验错误信息
	 * @param result 参数错误
	 * @return java.util.Map<java.lang.String,java.lang.String>
	 * @author wzy
	 * @date 2023/1/27 11:54
	 */
	private Map<String, String> getErrors(BindingResult result) {
		Map<String, String> map = new HashMap<>();
		List<FieldError> errorList = result.getFieldErrors();
		for (FieldError err : errorList) {
			// 发送验证错误的时候所对应的某个属性
			String field = err.getField();
			// 验证的错误消息
			String msg = err.getDefaultMessage();
			map.put(field, msg);
		}

		return map;
	}

//	private String getDefaultMsg(BindingResult result) {
//		String msg = null;
//
//		if(result == null || result.getAllErrors() == null
//				|| result.getAllErrors().size() == 0) return msg;
//
//		try {
//			msg = result.getAllErrors().get(0).getDefaultMessage();
//		} catch (Exception e) {}
//
//		return msg;
//	}

	/**
	 * 获得所有的异常信息
	 * @param result 异常
	 * @return java.lang.String
	 * @author wzy
	 * @date 2023/1/27 14:05
	 */
	private String getMsg(BindingResult result) {
		StringBuilder msg = new StringBuilder();
		List<FieldError> errorList = result.getFieldErrors();
		for (FieldError err : errorList) {
//			提示用户字段名称
			msg.append(err.getField());
			msg.append(":");
			msg.append(err.getDefaultMessage());
			msg.append(";");
		}
		return msg.toString();
	}


	/**
	 * 获得所有的异常信息
	 * @param e 异常
	 * @return java.lang.String
	 * @author wzy
	 * @date 2023/1/27 14:05
	 */
	private String getMsg(ConstraintViolationException e) {
		StringBuilder msg = new StringBuilder();
		for (ConstraintViolation<?> constraintViolation : e.getConstraintViolations()) {
			msg.append(constraintViolation.getMessage());
			msg.append(";");
		}
		return msg.toString();
	}

//	/**
//     * 全局异常捕捉处理
//     * @param ex
//     * @return
//     */
//	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
//    @ResponseBody
//    @ExceptionHandler(value = Exception.class)
//    public ApiResult exceptionHandler(Exception ex) {
//        ex.printStackTrace();
//        return ApiResult.errMsg(ex.getMessage());
//    }
}
