package com.peak.prd.base.service;

import com.peak.framework.common.BaseEntity;
import com.peak.framework.common.IEntityService;
import com.peak.framework.common.page.IQuery;
import com.peak.prd.base.dao.IPrdBaseDao;

import java.util.List;

/**
  * 产品Service基类；
  * 要优先使用批量调用，避免使用循环获取单个数据
 * @author zhangdexin
 */
public interface IPrdBaseService<E extends BaseEntity,PK,V extends IQuery<E>> extends IEntityService<E,PK,V> { 

	/** 获取缓存Entity（getById的缓存版本），缓存时效由外部配置文件确定 */
	public E findById(PK pk);
	
	/** 批量获取缓存Entities，返回结果集与ids顺序和数量严格一致，可能包含null元素； 要优先使用批量调用，避免使用循环获取单个数据 */
	public List<E> findByIds(List<PK> ids);
	
	/** 获取缓存VO */
	public <VO> VO findVOById(PK pk, Class<VO> clazz);

	/**
	 *
	 * @param pk
	 * @param clazz
	 * @param callCountService 是否调用 counterService.doCountFields
	 * @return
	 * @param <VO>
	 */
	public <VO> VO findVOById(PK pk, Class<VO> clazz, boolean callCountService);
	
	/** 批量获取缓存VOs，返回结果集与ids顺序和数量严格一致，可能包含null元素；要优先使用批量调用，避免使用循环获取单个数据 */
	public <VO> List<VO> findVOsByIds(List<PK> ids, Class<VO> clazz);

	/**
	 *
	 * @param ids
	 * @param clazz
	 * @param callCountService 是否调用 counterService.doCountFields
	 * @return
	 * @param <VO>
	 */
	public <VO> List<VO> findVOsByIds(List<PK> ids, Class<VO> clazz, boolean callCountService);
	
	/** getByIds：无cache */
	public List<E> getByIds(List<PK> ids);
	
	/** 
	 * 无cache<br/>
	 * 子类要自己实现本方法，底层findVOById和findVOByIds方法依赖此方法的实现<br/>
	 * <big><big><b>实现本方法时，千万不能调用自己的findVOById，否则会造成死循环</b></big><big><br/>
	 * 可以调用其它service的findVOById 
	 */
	public <VO> VO getVOById(PK pk, Class<VO> clazz);

	/** cacheName----------取值范围：PrdBaseService.CACHE_NAME_XXX---------- */
	public String getCacheName();
	
	//----------------------计数相关 ------------------
	/** 增加计数器
	 * @param pk id
	 * @param field 字段名
	 * @param delta 变化量
	 * @return 增加后的累计值
	 */
	public long increment(PK pk, String field, long delta);
	
	/**
	 * <p>向计算器以及计数排序set里添加数值</p>
	 * @param pk 元素ID 也是添加计数器increment方法的pk
	 * @param field increment方法的field 也是组成计数排序setkey的一部分 ,是业务数据的名称
	 * @param setPk setkey的开头 一般传level1domainId 意思是这个主域下一套排序  setPk+field 组成set的key
	 * @param delta 需要添加的变化量
	 * @return Double
	 * @since   1.0.0
	 * @author zhk
	 * @date 2022年1月21日 下午5:07:49 </p>
	 */
	public Double incrementSortedSet(PK pk, String field, String setPk, long delta);
	
	/**
	 * <p>获得排序值</p>
	 * @param setPk 一般传level1domainId 意思是这个主域下一套排序  setPk+field 组成set的key
	 * @param field 也是组成计数排序setkey的一部分 ,是业务数据的名称
	 * @param eneityId  元素ID
	 * @return Long
	 * @since   1.0.0
	 * @author zhk
	 * @date 2022年1月21日 下午5:18:40 </p>
	 */
	public Long getSortedSetRank(String setPk, String field, Long eneityId);
	
	/**
	 * 增加计数器 - 多计数字段
	 * @param pk id
	 * @param fields 字段名称数组
	 * @param deltas 变化量数组
	 * @return 增加后的累计值数组
	 */
	public long[] increment(PK pk, String[] fields, long[] deltas);
	
	/**
	 * 获得Dao对象，不再推荐使用
	 * @return
	 * @Deprecated 2022/8/25不再建议使用，Dao仅仅本service才能调用
	 * @date 2022/8/25 14:31
	 *
	 */
	@Deprecated
	public IPrdBaseDao<E,PK,V> getPrdEntityDao();
	/**
	 * 老的翻页方法，不再推荐使用
	 * @param query 查询条件
	 * @return
	 * @author wzy
	 * @Deprecated 2022/8/25不再建议使用，建议使用Dao层的findPaging
	 * @date 2022/8/25 14:31
	 *
	 */
	@Deprecated
	public V findPage(V query) ;

//	由于pagehelper与底层冲突，临时注释了
//	/**
//	 * 翻页转换方法
//	 * @param list PageHelper.startPage语句后第一个数据库查询返回的list
//	 * @param voList 返回给页面前端的volist
//	 * @return
//	 * @author wzy
//	 * @CreateDate 2022年1月8日
//	 *
//	 */
//	public <T, L> Paging<L> getPaging(List<T> list, List<L> voList);


	/**
	 * 执行插入操作，如果entity中值为null,那么对应的字段将不执行插入，这样就可以使用数据库中的默认值了<br/>
	 * <b>此方法要求数据库表必须有主键，支持组合主键，否则会报错</b>
	 * @param entity 插入的实体
	 * @return int 插入的行数
	 * @author wzy
	 * @date 2023/6/21 14:00
	 */
	int insertSelective(E entity);

	/**
	 * 老的插入方法，不再推荐使用
	 * @param entity 实体
	 * @return 更新的行数
	 * @author wzy
	 * @Deprecated 2023/6/21不再建议使用，建议使用insertSelective
	 * @date 2023/6/21 14:00
	 */
	@Deprecated
	public int insert(E entity);

	/**
	 * 执行清除缓存的统一方法
	 * @param id 关键Id
	 * @author wzy
	 * @date 2023/11/6 15:03
	 */
	void clearCache(Long id);

	/**
	 * @description 多参数清除缓存方法
 	 * @param values
	 * @return
	 * @author ZhengJia
	 * @date 2025/1/13 17:07
	 **/
	void clearCache(List<String> values);

}
