package com.peak.prd.base.service.imple;

import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.peak.common.util.IdUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.peak.common.util.ConvertUtil;
import com.peak.common.util.RedisService;
import com.peak.common.util.SpringUtil;
import com.peak.prd.base.model.CountSetting;
import com.peak.prd.base.service.IPrdBaseService;

/**
 * 计数器服务
 * @author zhangdexin
 */
@Slf4j
@Service
public class CounterService {
	@Autowired
	RedisService redisService;

	static final boolean _newFlag = true; // true优化后的代码 false老代码
	
	/** 处理计数字段 
	 * @param id id
	 * @param ev Entity 或者 VO
	 */
	public <PK, EV> void doCountFields(PK id, EV ev) {
		if (ev == null) return;
		List<CountSetting> countFields = CountSetting.match(ev.getClass().getSimpleName());
		if (countFields == null) return;

		if (_newFlag) {
			List<Long> counts = this.findCountsByIds(Arrays.asList(id), countFields);
			if (counts != null) {
				if (counts.size() != countFields.size()) {
					String gkey = IdUtil.getGkey();
					log.error("gkey:" + gkey + "---" + ev.getClass().getSimpleName() + "-countFields.size():" + countFields.size() + "---counts.size---" + counts.size());
					for (int i=0; i<countFields.size(); i++) {
						log.info(gkey + "---countFields[{}]--{}", i, countFields.get(i));
					}
				}

				for (int k=0; k<countFields.size(); k++) {
					CountSetting cs = countFields.get(k);
					Long countV = counts.get(k);
					if (countV != null) {
						ConvertUtil.setProperty(ev, cs.getField(), countV);
					}
				}
			}
		}
		else {
			for (CountSetting cs : countFields) {
				Long count = this.findCountById(id, cs); // 返回值可能是null
				ConvertUtil.setProperty(ev, cs.getField(), count);
			}
		}
	}
	
	/** 处理计数字段 
	 * @param id id
	 * @param ev Entity 或者 VO
	 */
	public <PK, EV> void doCountFields(List<PK> ids, List<EV> evs) {
		if (evs == null || evs.isEmpty()) return;
		if (evs.get(0) == null) return;
		List<CountSetting> countFields = CountSetting.match(evs.get(0).getClass().getSimpleName());
		if (countFields == null) return;

		if (_newFlag) {
			List<Long> counts = this.findCountsByIds(ids, countFields);
			if (counts != null && counts.size() > 0) {
				int index = 0;
				for (int j=0; j<ids.size(); j++) {
					for (int k=0; k<countFields.size(); k++) {
						CountSetting cs = countFields.get(k);
						Long countV = counts.get(index++);
						if (countV != null) {
							ConvertUtil.setProperty(evs.get(j), cs.getField(), countV);
						}
					}
				}
			}
		}
		else {
			for (CountSetting cs : countFields) {
				List<Long> counts = this.findCountsByIds(ids, cs); // 返回列表内的项可能是null
				for (int j=0; j<ids.size(); j++) {
					ConvertUtil.setProperty(evs.get(j), cs.getField(), counts.get(j));
				}
			}
		}
	}
	
	// TODO ： 优化
	public <E, PK> long increment(E entity, PK id, String field, long delta) {
		CountSetting cs = CountSetting.match(entity.getClass().getSimpleName(), field);
		if (cs == null) {
			throw new RuntimeException("CountSetting[" + entity.getClass().getSimpleName() + "." + field + "] not found");
		}
		String redisKey = getRedisKey(id, cs);
		long currCount = this.findCountById(id, cs) + delta;
		
		// 待优化: 异步处理
		redisService.increment(redisKey, delta);
		this.updateCount(entity.getClass(), id, currCount, cs); // 持久化
		
		return currCount;
	}
	
	/**
	 * <p>向计数排序set里添加数值</p>
	 * @param id   一般传level1domainId 意思是这个主域下一套排序  setPk+field 组成set的key
	 * @param field 业务数据的名称
	 * @param eneityId 元素ID
	 * @param delta 需要添加的变化量
	 * @return Double
	 * @since   1.0.0
	 * @author zhk
	 * @date 2022年1月21日 下午5:07:49 </p>
	 */
	public <E, PK> Double incrementSortedSet(PK id, String field, Long eneityId, long delta) {
	
		String redisKey = field + ":" + id;//获得排序组key 
		// 待优化: 异步处理
		return redisService.incrementZSetScore(redisKey, eneityId.toString(), delta);
	}
	
	
	public <E, PK> Long getSortedSetRank(PK id, String field, Long eneityId) {
		String redisKey = field + ":" + id;//获得排序组key 
		Long rank = redisService.getZSetReverseRank(redisKey, eneityId.toString());
		return rank==null?0:rank+1; //如果没有这个数据的排行返回0 ,如果有的话把redis返回值+1

	}
	

	/**
	 * 获得计数值（经过redis）
	 * @param counter
	 * @return
	 */
	private <PK> Long findCountById(PK id, CountSetting cs) {
		String redisKey = getRedisKey(id, cs);
		String count = redisService.get(redisKey);
		if (count == null) {
			// redis里还没有数据，那么首次从db拿，并且设置redis计数器初始值
			IPrdBaseService service = (IPrdBaseService)SpringUtil.getBean(cs.getEntityService());
			Long nCount = this.getCount(service.getById(id), cs.getEntityField());
			count = nCount == null ? "0" : nCount.toString(); // 初始值0
//			设置计数缓存过期时间为半小时
			redisService.setnx(redisKey, count, 1800); // setnx TODO
		}
		
		return Long.parseLong(count);
	}
	
	/** 批量获取计数数据 */
	private <PK> List<Long> findCountsByIds(List<PK> idList, CountSetting countField) {
		if (idList == null) return null;
		if (idList.isEmpty()) return new ArrayList<Long>();
		
		List<Long> values = new ArrayList<Long>();
		List<String> keys = this.getRedisKeys(idList, countField); // keys
		List<String> counts = redisService.multiGet(keys);  // values 可能含null
		for (int i=0; i<idList.size(); i++) {
			PK id = idList.get(i);
			String c = counts.get(i);
			if (c == null) {
				Long v = this.findCountById(id, countField);
				values.add(v); // 可能为null
			}
			else {
				values.add(new Long(c));
			}
		}
		return values;
	}

	/** 批量获取计数数据 */
	private <PK> List<Long> findCountsByIds(List<PK> idList, List<CountSetting> countFields) {
		if (idList == null || countFields == null) return null;

		int idsNum = idList.size();
		int countFieldNum = countFields.size();

		if (idsNum == 0 || countFieldNum == 0) return new ArrayList<Long>();

		List<Long> values = new ArrayList<Long>();
		List<String> keys = this.getRedisKeys(idList, countFields); // keys
		List<String> counts = redisService.multiGet(keys);  // values 可能含null

//		if (gkey != null) {
//			log.info("gkey:" + gkey + "---keys.size:" + keys.size() + "---mget.size:" + counts.size());
//		}

		int index = 0;

		for (int i=0; i<idsNum; i++) {
			PK id = idList.get(i);

			for (int j=0; j<countFieldNum; j++) {
				String c = counts.get(index++);
				if (c == null) {
					Long v = this.findCountById(id, countFields.get(j));
					values.add(v); // 可能为null
//					values.add(null);
				} else {
					values.add(new Long(c));
				}
			}
		}
		return values;
	}

	/** 在redis内的key */
	private <PK> List<String> getRedisKeys(List<PK> idList, List<CountSetting> countFields) {
		if (idList == null || countFields == null) return null;

		int idsNum = idList.size();
		int countFieldNum = countFields.size();

		if (idsNum == 0 || countFieldNum == 0) return null;

		List<String> keys = new ArrayList<String>();

		for (int i=0; i<idsNum; i++) {
			for (int j=0; j<countFieldNum; j++) {
				String str = getRedisKey(idList.get(i), countFields.get(j));
				keys.add(str);
			}
		}

		return keys;
	}
	
	/** 在redis内的key */
	private <PK> List<String> getRedisKeys(List<PK> idList, CountSetting countField) {
		if (idList == null) return null;
		
		List<String> keys = new ArrayList<String>();
		
		for (int i=0; i<idList.size(); i++) {
			String str = getRedisKey(idList.get(i), countField);
			keys.add(str);
		}
		return keys;
	}
	
	/** 在redis内的key */
	private <PK> String getRedisKey(PK id, CountSetting countSetting) {
		// 实体类名:计数字段名:id
		return countSetting.getEntityClassName() + ":" + countSetting.getEntityField() + ":" + id;
	}
	
	// 持久化
	private <PK> void updateCount(Class<?> entityClazz, PK id, Long countValue, CountSetting countSetting) {
		try {
			// 构造一个entity
			Object entity = entityClazz.newInstance();
			
			// set主键
			PropertyDescriptor idPropField = getPropertyDescriptor(entityClazz, countSetting.getEntityIdField());
			if (Integer.class.getTypeName().equals(idPropField.getPropertyType().getTypeName())) {
				idPropField.getWriteMethod().invoke(entity, new Integer(id.toString())); // propValue类型需要与set方法里的参数类型匹配	
			}
			else if (Long.class.getTypeName().equals(idPropField.getPropertyType().getTypeName())) {
				idPropField.getWriteMethod().invoke(entity, new Long(id.toString())); // propValue类型需要与set方法里的参数类型匹配	
			}
			else {
				throw new Exception("need int or long field");
			}
			
			// set计数字段
			PropertyDescriptor propField = getPropertyDescriptor(entityClazz, countSetting.getEntityField());
			if (Integer.class.getTypeName().equals(propField.getPropertyType().getTypeName())) {
				propField.getWriteMethod().invoke(entity, new Integer(countValue.toString())); // propValue类型需要与set方法里的参数类型匹配	
			}
			else if (Long.class.getTypeName().equals(propField.getPropertyType().getTypeName())) {
				propField.getWriteMethod().invoke(entity, new Long(countValue.toString())); // propValue类型需要与set方法里的参数类型匹配	
			}
			else if (Double.class.getTypeName().equals(propField.getPropertyType().getTypeName())) {
				propField.getWriteMethod().invoke(entity, new Double(countValue.toString())); // propValue类型需要与set方法里的参数类型匹配	
			}
			else {
				throw new Exception("===============need int or long or double field================");
			}
			
			// 持久化
			IPrdBaseService service = (IPrdBaseService)SpringUtil.getBean(countSetting.getEntityService());
			service.updateSelective(entity);
		} 
		catch (Exception ex) {
//			log.error("updateCount-error:[countSetting.getEntityService():" + countSetting.getEntityService() + "][error:" + ex.getMessage() + "]");
			log.error("updateCount-error:" + ex.getMessage());
			log.error("updateCount-error-countSetting.getEntityService:" + countSetting.getEntityService());
			throw new RuntimeException(ex);
		}
	}
	
	// 反射获取count字段值，可能返回null
	private Long getCount(Object entity, String propName) {
		if (entity == null) return null;
		
		try {
			PropertyDescriptor propField = getPropertyDescriptor(entity.getClass(), propName);
			Object propVal = propField.getReadMethod().invoke(entity);
			if (propVal == null) return null;
			
			String str = propVal.toString();
			int pos = str.indexOf(".");
			if (pos >= 0) {
				str = str.substring(0, pos);
			}
			if (str.isEmpty()) return 0L;
			
			return new Long(str);
		}
		catch (Exception ex) {
			throw new RuntimeException(ex);
		}
	}
	
	// 获取bean的getter/setter属性
	private static PropertyDescriptor getPropertyDescriptor(Class<?> clazz, String name) throws IntrospectionException {
		for (PropertyDescriptor propertyDescriptor : Introspector.getBeanInfo(clazz).getPropertyDescriptors()) {
			if (propertyDescriptor.getName().equals(name)) {
				return propertyDescriptor;
			}
		}
		
		return null;
	}
	
}
