package com.peak.prd.base.service.imple;

import com.peak.common.util.SpringUtil;
import com.peak.framework.common.BaseEntity;
import com.peak.framework.common.BaseService;
import com.peak.framework.common.bean.BeanUtils;
import com.peak.framework.common.page.IQuery;
import com.peak.prd.base.dao.IPrdBaseDao;
import com.peak.prd.base.service.IDataAuthService;
import com.peak.prd.base.service.IDomainAuthService;
import com.peak.prd.base.service.IPrdBaseService;
import com.peak.prd.exception.DataAuthException;
import com.peak.prd.exception.PrepareException;
import io.netty.util.internal.StringUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * 产品Service基类
 * @author zhangdexin
 */

@Service
public abstract class PrdBaseService <E extends BaseEntity,PK,V extends IQuery<E>> extends BaseService<E,PK,V> implements IPrdBaseService<E,PK,V>, IDataAuthService<E> {
	
	@Autowired
	PrdCacheService prdCacheService;
	
	@Autowired
	CounterService counterService;
	
	@Autowired
	com.peak.prd.config.CacheConfig cacheConfig;
	
	public static final String CACHE_NAME_PRD = "prd"; // default
	public static final String CACHE_NAME_60  = "cache60";
	
	/** 获取缓存Entity（getById的缓存版本），缓存时效由外部配置文件确定 */
	public E findById(PK pk) {
		if (pk == null) return null;
		
		E entity = null;
		if (CACHE_NAME_60.equals(this.getCacheName())) {
			if (cacheConfig.isRedisCache()) {
				entity = prdCacheService.findById_RedisCache60(pk, this);
			}
			else {
				entity = prdCacheService.findById_EhcacheCache60(pk, this);
			}	
		}
		else {
			entity = prdCacheService.findById(pk, this);
		}
		
		// 处理计数字段
		counterService.doCountFields(pk, entity);
		
		return entity;
	}
	
	/** 批量获取缓存Entities，返回结果集与ids顺序和数量严格一致，可能包含null元素 */
	public List<E> findByIds(List<PK> ids) {
		if (ids == null || ids.isEmpty()) return null;
		List<E> entities = prdCacheService.findByIds(ids, this);
		
		// 处理计数字段s
		counterService.doCountFields(ids, entities);
				
		return entities;
	}
	
	/** 获取缓存VO（getVOById的缓存版本） */
	@Override
	public <VO> VO findVOById(PK pk, Class<VO> clazz) {
		return this.findVOById(pk, clazz, true);
	}

	@Override
	public <VO> VO findVOById(PK pk, Class<VO> clazz, boolean callCountService_del) {
		if (pk == null) return null;
		VO vo = null;
		
		if (CACHE_NAME_60.equals(this.getCacheName())) {
			if (cacheConfig.isRedisCache()) {
				vo = prdCacheService.findVOById_RedisCache60(pk, clazz, this);
			}
			else { 
				vo = prdCacheService.findVOById_EhcacheCache60(pk, clazz, this);
			}
		}
		else {
			vo = prdCacheService.findVOById(pk, clazz, this);
		}
		
		// 处理计数字段s
		if (true) {
			counterService.doCountFields(pk, vo);
		}

		return vo;
	}
	
	/** 批量获取缓存VOs，返回结果集与ids顺序和数量严格一致，可能包含null元素 */
	@Override
	public <VO> List<VO> findVOsByIds(List<PK> ids, Class<VO> clazz) {
		return this.findVOsByIds(ids, clazz, true);
	}

	@Override
	public <VO> List<VO> findVOsByIds(List<PK> ids, Class<VO> clazz, boolean callCountService_del) {
		if (ids == null || ids.isEmpty()) return null;
		List<VO> vos = prdCacheService.findVOsByIds(ids, clazz, this);
		
		// 处理计数字段s
		if (true) {
			counterService.doCountFields(ids, vos);
		}

		return vos;
	}
	
	/** getByIds：无cache */
	public List<E> getByIds(List<PK> ids) {
		return ((IPrdBaseDao)getEntityDao()).getByIds(ids);
	}
	
	/** 子类要自己实现本方法------子类不能调用自己的findVOById-------子类可以调用其他service的findVOById */
	public <VO> VO getVOById(PK pk, Class<VO> clazz) {
		// 有需要的子类应该覆盖实现本方法
		// 本方法实现代码不必刻意使用缓存
		throw new RuntimeException(this.getClass().getName() + "需要实现getVOById(...)方法");
	}
	
	/** cacheName */
	public String getCacheName() {
		return CACHE_NAME_PRD; // default
	}

	//----------------------计数相关 ------------------
	public long increment(PK pk, String field, long delta) {
		E entity = this.findById(pk);
		if (entity == null) return 0L; // 一般来说，不应该执行到这里

		return counterService.increment(entity, pk, field, delta);
	}
	
	public long[] increment(PK pk, String[] fields, long[] deltas) {
		// TODO 优化
		if (fields == null || deltas == null || fields.length == 0) throw new RuntimeException("invalid increment args[a]");
		if (fields.length != deltas.length) throw new RuntimeException("invalid increment args[b]");
		
		long[] ret = new long[fields.length];
		ret[0] = this.increment(pk, fields[0], deltas[0]);
		ret[1] = this.increment(pk, fields[1], deltas[1]);
		
		return ret;
	}
	
	
	public Double incrementSortedSet(PK pk, String field, String setPk, long delta) {
		if ((pk == null)||StringUtil.isNullOrEmpty(field)) return 0D; // 一般来说，不应该执行到这里
		//向计数器添加数值
		this.increment(pk, field, delta);
		//向计数排序set里添加数值
		return counterService.incrementSortedSet(setPk, field, (Long)pk, delta);
	}
	
	public Long getSortedSetRank(String setPk, String field, Long eneityId) {
		if (StringUtil.isNullOrEmpty(setPk)||StringUtil.isNullOrEmpty(field)) return 0L; // 一般来说，不应该执行到这里
		
		return counterService.getSortedSetRank(setPk, field,eneityId);
	}
	/**
	 * 获得Dao对象，不再推荐使用
	 * @return
	 * @Deprecated 2022/8/25不再建议使用，Dao仅仅本service才能调用
	 * @date 2022/8/25 14:31
	 *
	 */
	@Deprecated
	public IPrdBaseDao<E,PK,V> getPrdEntityDao() {
		return (IPrdBaseDao<E,PK,V>)this.getEntityDao();
	}
	/**
	 * 老的翻页方法，不再推荐使用
	 * @param query 查询条件
	 * @return
	 * @author wzy
	 * @Deprecated 2022/8/25不再建议使用，建议使用Dao层的findPaging
	 * @date 2022/8/25 14:31
	 *
	 */
	@Override
	@Deprecated
	public V findPage(V query) {
		return super.findPage(query);
	}

	//	由于pagehelper与底层冲突，临时注释了
//	@Override
//	public <T, L> Paging<L> getPaging(List<T> list,List<L> voList) {
//
//		PageInfo<T> pageInfo = new PageInfo<T>(list);
//		
//		Paging<L> paging = new Paging<L>(pageInfo.getPageNum(), pageInfo.getPageSize());
//		 
//		paging.setTotalCount(pageInfo.getTotal());
//		paging.setList(voList);
//		return paging;
//	}
		
	@Override
	public E dataAuth(Long entityId, Long currUserId, Long curMngDomainId) {
		if (entityId == null) {
			throw new PrepareException("数据未找到[" + entityId + "]");
		}

		E entity = this.getById((PK)entityId);
		if (entity == null) {
			throw new PrepareException("数据未找到[" + entityId + "]");
		}

		// 数据越权检查 反射 拿到domainId
		String strDomainId = BeanUtils.getProperty(entity, "domainId");
		
		if (strDomainId != null) {
			IDomainAuthService domainAuthService = SpringUtil.getBean(IDomainAuthService.class);
			if (!domainAuthService.domainAuth(new Long(strDomainId), currUserId, curMngDomainId)) {
				throw new DataAuthException("数据访问越权[entityId:" + entityId + ", domainIdsOfData:" + strDomainId 
						+ ", currUserId:" + currUserId + ", currMngDomainId:" + curMngDomainId + "]");
			}
		}
		
		return entity;
	}
	
	@Override
	public List<E> dataAuth(List<Long> entityIds, Long currUserId, Long curMngDomainId) {
		if (entityIds == null || entityIds.isEmpty()) {
			throw new PrepareException("数据未找到[" + entityIds + "]");
		}
		
		// list转成set  去重
		Set<PK> idSet = new HashSet<PK>();
		for (int i=0; i<entityIds.size(); i++) {
			idSet.add((PK)entityIds.get(i));
		}
		
		List<PK> idList = new ArrayList<PK>(idSet); // 去重后的list
		
		List<E> entities = this.getByIds(idList);
		
		if (entities == null || entities.isEmpty()) {
			throw new PrepareException("数据未找到[" + entityIds + "]");
		}
		
		// 数据越权检查 反射 拿到domainId  --- 循环
		List<Long> domainIdsOfData = new ArrayList<Long>();
		for (int i=0; i<entities.size(); i++) {
			String strDomainId = BeanUtils.getProperty(entities.get(i), "domainId");
			if (strDomainId != null) {
				domainIdsOfData.add(new Long(strDomainId));
			}
		}
		
		IDomainAuthService domainAuthService = SpringUtil.getBean(IDomainAuthService.class);
		if (!domainAuthService.domainAuth(domainIdsOfData, currUserId, curMngDomainId)) {
			throw new DataAuthException("数据访问越权[entityIds:" + entityIds + ", domainIdsOfData:" + domainIdsOfData 
					+ ", currUserId:" + currUserId + ", currMngDomainId:" + curMngDomainId + "]");
		}
		
		return entities;
	}

	@Override
	public int insertSelective(E entity) {
		return ((IPrdBaseDao)this.getEntityDao()).insertSelective(entity);
	}


	@Override
	public int insert(E entity) {
		return super.insert(entity);
	}

	@Override
	public void clearCache(Long id) {
		// 有需要的子类应该覆盖实现本方法
		throw new RuntimeException(this.getClass().getName() + "需要实现clearCache(id)方法");
	}

	@Override
	public void clearCache(List<String> values) {
		// 有需要的子类应该覆盖实现本方法
		throw new RuntimeException(this.getClass().getName() + "需要实现clearCache(values)方法");
	}

}
