package com.peak.prd.base.service.imple;

import com.peak.common.util.RedisObjectService;
import com.peak.framework.common.BaseEntity;
import com.peak.framework.common.page.IQuery;
import com.peak.prd.base.service.IPrdBaseService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.support.NullValue;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

/**
 * 产品Service基类
 * @author zhangdexin
 */

@Slf4j
@Service
public class PrdCacheService {
	@Autowired
	com.peak.prd.config.CacheConfig cacheConfig;

	@Autowired
	RedisObjectService redisObjectService;

	private AtomicLong findVOsByBatch_callNum = new AtomicLong();
	private AtomicLong findVOsByBatch_findVoById_callNum = new AtomicLong();
	private AtomicLong findVOById_callNum = new AtomicLong();

	/** 获取缓存Entity(redis|ehcache) */
	@Cacheable(key="#root.args[1].class.simpleName + ':' + #root.args[0]", cacheNames = "prd", cacheManager="cacheManager")
	public <E extends BaseEntity,PK,V extends IQuery<E>> E findById(PK pk, IPrdBaseService<E,PK,V> service) {
		return service.getById(pk);
	}

	/** 获取缓存VO(redis|ehcache) */
	@Cacheable(key="#root.args[2].class.simpleName + ':' + #root.args[1].simpleName + ':' + #root.args[0]", cacheNames = "prd", cacheManager="cacheManager")
	public <E extends BaseEntity,PK,V extends IQuery<E>,VO> VO findVOById(PK pk, Class<VO> clazz, IPrdBaseService<E,PK,V> service) {
		this.findVOById_callNum.incrementAndGet();
		return service.getVOById(pk, clazz);
	}

	/** 获取缓存Entity */
	@Cacheable(key="#root.args[1].class.simpleName + ':' + #root.args[0]", cacheNames = "prd", cacheManager="redisCache60")
	public <E extends BaseEntity,PK,V extends IQuery<E>> E findById_RedisCache60(PK pk, IPrdBaseService<E,PK,V> service) {
		return service.getById(pk);
	}

	/** 获取缓存VO */
	@Cacheable(key="#root.args[2].class.simpleName + ':' + #root.args[1].simpleName + ':' + #root.args[0]", cacheNames = "prd", cacheManager="redisCache60")
	public <E extends BaseEntity,PK,V extends IQuery<E>,VO> VO findVOById_RedisCache60(PK pk, Class<VO> clazz, IPrdBaseService<E,PK,V> service) {
		return service.getVOById(pk, clazz);
	}

	/** 获取缓存Entity */
	@Cacheable(key="#root.args[1].class.simpleName + ':' + #root.args[0]", cacheNames = "cache60")
	public <E extends BaseEntity,PK,V extends IQuery<E>> E findById_EhcacheCache60(PK pk, IPrdBaseService<E,PK,V> service) {
		return service.getById(pk);
	}

	/** 获取缓存VO */
	@Cacheable(key="#root.args[2].class.simpleName + ':' + #root.args[1].simpleName + ':' + #root.args[0]", cacheNames = "cache60")
	public <E extends BaseEntity,PK,V extends IQuery<E>,VO> VO findVOById_EhcacheCache60(PK pk, Class<VO> clazz, IPrdBaseService<E,PK,V> service) {
		return service.getVOById(pk, clazz);
	}

	/** 批量获取缓存Entities，返回结果集与ids顺序和数量严格一致，可能包含null元素 */
	public <E extends BaseEntity,PK,V extends IQuery<E>> List<E> findByIds(List<PK> ids, IPrdBaseService<E,PK,V> service) {
		if (cacheConfig.isRedisCache()) {
			// 批量
			return findEntitiesByBatch(ids, service);
		}

		// 轮询
		return findEntitiesByLoop(ids, service);
	}

	/** 批量 */
	private <E extends BaseEntity,PK,V extends IQuery<E>> List<E> findEntitiesByBatch(List<PK> ids, IPrdBaseService<E,PK,V> service) {
		List<String> keys = new ArrayList<String>();
		for (PK id : ids) {
			String key = "prd::" + service.getClass().getSimpleName() + ":" + id; // spring cache redis key
			keys.add(key);
		}

		// 使用redis mget批量获取缓存数据，返回必定不为null
		List entities = redisObjectService.multiGet(keys);

		// 批量返回的结果，可能不完全，需要做补充操作
		for (int i=0; i<entities.size(); i++) {
			boolean isNull = (entities.get(i) == null) || (entities.get(i) instanceof NullValue); // spring cache 用 NullValue对象表示null存到redis里
			if (!isNull)
				continue;

			if (entities.get(i) instanceof NullValue) {
				// 已缓存null
				entities.set(i, null);
			}
			else {
				// TODO getByIds
				// 未缓存
				E e = (E)service.findById(ids.get(i)); // 不可以直接调用this，那样不会走缓存
				entities.set(i, e); // 可能null
			}
		}

		return entities;
	}

	/** 轮询获取Entities */
	private <E extends BaseEntity,PK,V extends IQuery<E>> List<E> findEntitiesByLoop(List<PK> ids, IPrdBaseService<E,PK,V> service) {
		List<E> list = new ArrayList<E>();
		for (PK id : ids) {
			E e = (E)service.findById(id); // 不可以直接调用this，那样不会走缓存
			list.add(e);
		}
		return list;
	}

	/** 批量获取缓存VOs，返回结果集与ids顺序和数量严格一致，可能包含null元素 */
	public <E extends BaseEntity,PK,V extends IQuery<E>, VO> List<VO> findVOsByIds(List<PK> ids, Class<VO> clazz, IPrdBaseService<E,PK,V> service) {
		if (cacheConfig.isRedisCache()) {
			// 批量
			return findVOsByBatch(ids, clazz, service);
		}

		// 轮询
		return findVOsByLoop(ids, clazz, service);
	}

	/** 批量获取VOs */
	private <E extends BaseEntity,PK,V extends IQuery<E>, VO> List<VO> findVOsByBatch(List<PK> ids, Class<VO> clazz, IPrdBaseService<E,PK,V> service) {
		this.findVOsByBatch_callNum.incrementAndGet();

		List<String> keys = new ArrayList<String>();
		for (PK id : ids) {
//			String key = "prd::" + clazz.getName() + ":" + id; // spring cache redis key
			String key = "prd::" + service.getClass().getSimpleName() + ":" + clazz.getSimpleName() + ":" + id; // spring cache redis key
			keys.add(key);
		}

		// 使用redis mget批量获取缓存数据，返回必定不为null
		List<VO> vos = (List<VO>)redisObjectService.multiGet(keys);

		// 批量返回的结果，可能不完全，需要做补充操作
		for (int i=0; i<vos.size(); i++) {
			boolean isNull = (vos.get(i) == null) || (vos.get(i) instanceof NullValue); // spring cache 用 NullValue对象表示null存到redis里
			if (!isNull)
				continue;

			if (vos.get(i) instanceof NullValue) {
				// 已缓存null
				vos.set(i, null);
			}
			else {
				// 未缓存
				this.findVOsByBatch_findVoById_callNum.incrementAndGet();
				VO vo = (VO)service.findVOById(ids.get(i), clazz); // 不可以直接调用this，那样不会走缓存
				vos.set(i, vo); // 可能null
			}
		}

		return vos;
	}

	/** 轮询获取VOs */
	private <E extends BaseEntity,PK,V extends IQuery<E>, VO> List<VO> findVOsByLoop(List<PK> ids, Class<VO> clazz, IPrdBaseService<E,PK,V> service) {
		List<VO> list = new ArrayList<VO>();
		for (PK id : ids) {
			VO vo = (VO)service.findVOById(id, clazz); // 不可以直接调用this，那样不会走缓存
			list.add(vo);
		}
		return list;
	}

	/**
	 * 仅调试用，无业务实际用途，返回内容随时可能调整
	 * @return
	 */
	public String getDebugInfo() {
		return this.findVOById_callNum.get()
				+ "_"
				+ this.findVOsByBatch_callNum.get()
				+ "_"
				+ this.findVOsByBatch_findVoById_callNum.get();
	}

	private void clearDebugInfo() {
		this.findVOById_callNum.set(0L);
		this.findVOsByBatch_callNum.set(0L);
		this.findVOsByBatch_findVoById_callNum.set(0L);
	}

	public String resetDebugInfo() {
		this.clearDebugInfo();
		return this.getDebugInfo();
	}

}
