package com.peak.prd.base.util;

import com.peak.common.util.MD5Util;
import com.peak.common.util.Sm3Util;
import com.peak.common.util.SpringUtil;
import com.peak.prd.base.model.Paging;
import com.peak.prd.base.service.IOssService;
import com.peak.prd.config.AppCommonConfig;
import lombok.extern.slf4j.Slf4j;

import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

/**
 * 拼接图片绝对路径用
 * @author zhangdexin
 *
 */
@Slf4j
public class ResPathUtil {
	/**
	 * 拼接prefixPath+资源相对路径==>全路径
	 * @param entity 要处理的资源对象s
	 * @param fieldFullPath 属性的名称（大小写要完全匹配）（支持多级属性）
	 * @param prefixPath 要拼接的前缀路径
	 */
	public static Object prependPath(Object entity, String fieldFullPath, String prefixPath) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException, IntrospectionException {
		return prependPath(entity,fieldFullPath,prefixPath,false);
	}
	/**
	 * 拼接prefixPath+资源相对路径==>全路径 
	 * @param entity 要处理的资源对象s
	 * @param fieldFullPath 属性的名称（大小写要完全匹配）（支持多级属性）
	 * @param prefixPath 要拼接的前缀路径
	 * @param islessonpath 是否课程路径，课程路径http全路径时，不进行处理，其他都需要截取掉
	 */
	public static Object prependPath(Object entity, String fieldFullPath, String prefixPath, boolean islessonpath) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException, IntrospectionException {
		if (entity == null || fieldFullPath == null || fieldFullPath.isEmpty() || prefixPath == null || prefixPath.isEmpty()) {
			return entity;
		}
		
		List leafs = getLeafs(entity, fieldFullPath); // 寻找叶子节点，最终拼接都是在叶子节点上

		if (leafs == null || leafs.isEmpty()) return entity;
		
		String lastFieldName = fieldFullPath.substring(fieldFullPath.lastIndexOf(".") + 1); // 最后一级属性名称
		PropertyDescriptor pd = getPropertyDescriptor(leafs.get(0).getClass(), lastFieldName); // 属性描述符
		AppCommonConfig config = SpringUtil.getBean(AppCommonConfig.class);
		// 叶子对象集合
		for (int i=0; i<leafs.size(); i++) { // 经测试，轮询读取、写入速度都是很快的
			Object relativePath = pd.getReadMethod().invoke(leafs.get(i)); // 读取相对路径
						
			if (relativePath == null || relativePath.toString().isEmpty())	continue;

//			filename=不拼上prefixPath的原因是读流时需要前端拼上服务名称
			if (relativePath.toString().indexOf("?filename=") > -1) {
				continue;
			}
//			wzy 2021年10月29日 添加if判断如果relativePath是以http或https开头的全路径，则无需重写
//			因为当从分布式内容服务器读取课件时，返回的就是带tip的全路径，不能再拼上prefixPath了
//			增加判断只有课程时才不拼接，因为openfeign调用时也会返回带http的路径，此路径需要剔除掉
			if (relativePath.toString().startsWith("http") && islessonpath) {
				continue;
			}
			
//			wzy 2021年12月18日 要求替换的属性中如果是list 如：private List<String> imglist;
			if (relativePath instanceof List) {
				//为list中每一个值拼加前缀，形成全路径
				for (int j=0; j < ((List)relativePath).size(); j++) {
					Object obj =((List) relativePath).get(j);
					//如果存在http开头全路径url，截取到第一个"/"前，在openfeign调用时也会返回带http的路径，此路径需要剔除掉
					obj = substringDomainname(obj);

					if (obj != null) {
						//启用oss
						if (config.getOssEnable()) {
							IOssService ossService = (IOssService)SpringUtil.getBean("ossService");
							String objUrl = (String)obj;
							String url = ossService.getUrl(objUrl);//通过oss获得访问路径，若不使用oss返回传参
							((List) relativePath).set(j, url);// 写入oss全路径
						} else{
							//未启用oss
							obj = prefixPath + obj;
							((List) relativePath).set(j, obj); // 写入全路径
						}

					}
				}
				pd.getWriteMethod().invoke(leafs.get(i), protectUrlList((List)relativePath)); // 写回list属性
			} else {
				//如果存在http开头全路径url，截取到第一个"/"前，在openfeign调用时也会返回带http的路径，此路径需要剔除掉
				relativePath = substringDomainname(relativePath);
				//启动oss
				if (config.getOssEnable()) {
					IOssService ossService = (IOssService)SpringUtil.getBean("ossService");
					String objUrl = (String)relativePath;
					String url = ossService.getUrl(objUrl);//通过oss获得访问路径，若不使用oss返回传参
					pd.getWriteMethod().invoke(leafs.get(i), protectUrl(url)); // 写入oss全路径
				} else {
					//未启用oss
					String fullpath = prefixPath + relativePath;
					pd.getWriteMethod().invoke(leafs.get(i), protectUrl(fullpath)); // 写入全路径
				}
			}
			
		}
		

		return entity;
	}

	/**
	 * list保护
	 * @param list
	 * @return
	 */
	private static List protectUrlList(List list) {
		for (int i=0; i<list.size(); i++) {
			Object url = list.get(i);
			url = protectUrl(url);
			list.set(i, url);
		}

		return list;
	}

	/**
	 * 对url补充防盗属性（签名+时间戳）
	 * @param oUrl 原始url数据
	 * @return 保护后的url
	 * @author zhangdexin
	 * @date 2024-09-13
	 */
	private static Object protectUrl(Object oUrl) {
		if (oUrl == null) {
			return null;
		}

		AppCommonConfig appCommonConfig = SpringUtil.getBean(AppCommonConfig.class);
		if (appCommonConfig.getRespathProtectEnable() != 1) {
			return oUrl; // 不防盗
		}

		String sUrl = oUrl.toString();
		if (sUrl.isEmpty()) {
			return oUrl;
		}

		URL url;
		try {
			url = new URL(sUrl);
		} catch (MalformedURLException e) {
			log.warn("protectUrl fail[url:{}][err:{}], ", sUrl, e.getMessage());

			return oUrl;
		}

		String path = url.getPath(); // 以`/`开头，不含query
		String tsFormat = appCommonConfig.getRespathProtectTimestampFormat();        // 时间戳格式定义
		String timestamp = (tsFormat == null || tsFormat.isEmpty())
				? Long.toString(System.currentTimeMillis())
				: DateTimeFormatter.ofPattern(tsFormat).format(LocalDateTime.now()); // 时间戳

		sUrl += (sUrl.indexOf("?") < 0) ? "?" : "&";
		sUrl += appCommonConfig.getRespathProtectSignName() + "="
				+ getSign(appCommonConfig.getRespathProtectAlgorithm(), appCommonConfig.getRespathProtectSeckey(), appCommonConfig.getRespathProtectSignFormat(), path, timestamp);
		sUrl += "&" + appCommonConfig.getRespathProtectTimestampName() + "=" + timestamp;

		return sUrl;
	}

	/**
	 * 获取URL路径数据的签名(hash)
	 * https://btool.cn/hmac-generator 在线sm3加密
	 * https://md5jiami.bmcx.com/#google_vignette 在线md5
	 * @param alg 算法名称
	 * @param seckey 密钥
	 * @param signFormat 签名格式
	 * @param path 路径
	 * @param timestamp 时间戳
	 * @return 签名结果
	 * @author zhangdexin
	 * @date 2024-09-13
	 */
	public static String getSign(String alg, String seckey, String signFormat, String path, String timestamp) {
		String mTxt; // 签名结果
		String text = signFormat.replace("$seckey", seckey).replace("$timestamp", timestamp).replace("$path", path); // 明文

		if ("md5".equalsIgnoreCase(alg)) {
			mTxt = MD5Util.md5Encrypt(text);
		}
		else { // 缺省 == sm3
			mTxt = Sm3Util.hmacToString(seckey.getBytes(), text); // sm3 国密hash算法（类似md5） hex string
			//		明文：123/static/hello1.jpg1725933146005---END--- // 密钥123 时间戳1725933146005
			//		密文：81e59884898a8c4c93cbb4978a96cc83f2bc0e308bca21bd3f170cab51e2c1c2---END---
		}

		if (log.isDebugEnabled()) {
			log.debug("alg: {}, text: {}, mTxt: {}", alg
					, text.replace(seckey, "***")
					, mTxt);
		}

		return mTxt;
	}

	/**
	 * 如果存在域名将域名截取掉
	 * @param obj
	 * @return 截取掉域名后的url路径
	 * @author wzy
	 * @date 2024/1/5 14:08
	 */
	private static Object substringDomainname(Object obj) {
		if (obj == null) {
			return null;
		}
		String objStr = obj.toString();
		if (objStr.startsWith("http://")) {
			objStr = objStr.substring(objStr.indexOf("http://")+7);
			if (objStr.indexOf("/") > -1) {
				objStr = objStr.substring(objStr.indexOf("/")+1);
			}

		 } else if (objStr.toString().startsWith("https://")) {
			objStr = objStr.substring(objStr.indexOf("https://")+8);
			if (objStr.indexOf("/") > -1) {
				objStr = objStr.substring(objStr.indexOf("/")+1);
			}
		}

		return objStr;
	}


	private static List<?> getLeafs(Object entity, String fieldFullPath) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException, IntrospectionException {
		if (entity instanceof List) return getLeafsByList((List)entity, fieldFullPath); 
		if (entity instanceof Paging) {
			Paging page = (Paging)entity;
			if (page.getList() != null) {
				return getLeafsByList(page.getList(), fieldFullPath); 
			}
		}
		return getLeafsByObject(entity, fieldFullPath);
	}
	
	private static List<?> getLeafsByList(List srcList, String fieldFullName) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException, IntrospectionException {
		if (srcList == null || srcList.isEmpty()) return srcList;
		
		// 已经是叶子级
		if (fieldFullName.indexOf(".") < 0) return srcList; 
		
		List list = new ArrayList();
		
		for (int i=0; i<srcList.size(); i++) {
			Object item = srcList.get(i);
			List leafs = getLeafs(item, fieldFullName);
			if (leafs != null && !leafs.isEmpty()) {
				list.addAll(leafs);	
			}
		}
		
		return list;
	}
	
	private static List<?> getLeafsByObject(Object srcObj, String fieldFullName) throws IntrospectionException, IllegalAccessException, IllegalArgumentException, InvocationTargetException {
		if (srcObj == null) return null;
		
		// 已经是叶子级
		if (fieldFullName.indexOf(".") < 0) {
			List list = new ArrayList();
			list.add(srcObj);
			return list;
		}
		
		List list = new ArrayList();
		// 当前不是叶子级别，有下级要处理
		String currPropName = fieldFullName.substring(0, fieldFullName.indexOf(".")); // 当前级属性
		PropertyDescriptor currPd = getPropertyDescriptor(srcObj.getClass(), currPropName); // 当前级属性描述符
//		wzy 2021年11月16日增加判断，因为在某些VO中仅仅是部分子VO属性才有对应的属性，例如TaUsercontentpassVO的TaUsercontentVO中包含多个子VO，每次仅给1个VO赋值，导致有时获取不到对应的属性
		if (currPd == null) {
			return null;
		}
		Object propValue = currPd.getReadMethod().invoke(srcObj); // 读取属性
		if (propValue instanceof List) {
			// 属性值是列表
			List tmpList = (List)propValue;
			for (int i=0; i<tmpList.size(); i++) {
				List leafs = getLeafs(tmpList.get(i), fieldFullName.substring(currPropName.length() + 1)); // 下级属性
				if (leafs != null && !leafs.isEmpty()) {
					list.addAll(leafs);	
				}
			}
		}
		else if (propValue != null) {
			// 属性值是一般对象
			List leafs = getLeafs(propValue, fieldFullName.substring(currPropName.length() + 1)); // 下级属性
			if (leafs != null && !leafs.isEmpty()) {
				list.addAll(leafs);	
			}
		}
		
		return list;
	}
	
	// 获取bean的getter/setter属性
	private static PropertyDescriptor getPropertyDescriptor(Class clazz, String name) throws IntrospectionException {
		for (PropertyDescriptor propertyDescriptor : Introspector.getBeanInfo(clazz).getPropertyDescriptors()) {
			if (propertyDescriptor.getName().equals(name)) {
				return propertyDescriptor;
			}
		}
		
		return null;
	}
}
