package com.peak.prd.quartz.lock;

import java.util.List;

import com.peak.common.util.RedisService;
import com.peak.common.util.SpringUtil;
import com.peak.prd.config.JobConfig;
import com.peak.prd.quartz.health.Health;
import com.peak.prd.quartz.health.HealthHelper;

/**
 * job服务多机多实例，禁止同时执行
 * @author zhangdexin
 *
 */
public class JobLockUtil {

	/**
	 * job方法加锁
	 * @param methodFullname
	 * @return true加锁成功  false加锁失败
	 */
	public static boolean lock(String methodFullname) {
		RedisService redisService = SpringUtil.getBean(RedisService.class);
		String lockName = getLocalLockKey(methodFullname);     // 锁名
		String lockVal  = innerLock(lockName);
		if (lockVal == null) {
			return true;  // 加锁成功
		}
		
		if (isAlive(lockVal)) {
			return false; // 加锁失败：被其他活跃实例加锁了
		}
		else {
			// 前一个加锁服务已经死了（比如手工重启等）
			// 强制解锁
			redisService.delLock(lockName);
		}

		// 再次尝试加锁
		return innerLock(lockName) == null;
	}
	
	/**
	 * 加锁成功返回`null`，加锁失败返回`当前锁值`
	 * <p>方法名称全路径，用于`锁名`</p>
	 * <p>服务运行实例id，用于`锁值`</p>
	 */
	private static String innerLock(String lockName) {
		RedisService redisService = SpringUtil.getBean(RedisService.class);
		String srvInstanceUuid = getServerInstanceUuid(); // 锁值：服务运行实例id

		Boolean b = redisService.lock(lockName, srvInstanceUuid, Integer.MAX_VALUE); // 无限期加锁
		if (b != null && b.booleanValue()) {
			return null; // 加锁成功
		}
		
		String lockVal = redisService.getLockValue(lockName);
		if (srvInstanceUuid.equals(lockVal)) {
			return null; // 加锁成功：本服务之前加的
		}
		
		return lockVal;  // 加锁失败：其他服务的uuid
	}
	
	/** job方法解锁 */
	public static void unlock(String methodFullname) {
		RedisService redisService = SpringUtil.getBean(RedisService.class);
		
		redisService.unlock(getLocalLockKey(methodFullname), getServerInstanceUuid());
	}
	
	/**
	 * 锁的名称（不是redis的真实key） ， 不同的名称 表示不同的锁 ，相同的名称 表示同一个锁
	 * @param methodFullname
	 * @return
	 */
	private static String getLocalLockKey(String methodFullname) {
		return "job:lock:" + methodFullname;
	}

	/**
	 * 服务运行实例id，每次启动都不一样
	 * <p>用于lock value</p>
	 */
	private static String getServerInstanceUuid() {
		return SpringUtil.getBean(JobConfig.class).getUuid();
	}
	
	/**
	 * 判断前一个加锁服务是否还活着，如果服务已经死了，则就忽略该服务已经加了的锁
	 * @param uuid 前一个加锁服务的唯一运行id
	 * @return true活着  false死了
	 */
	private static boolean isAlive(String uuid) {
		List<Health> healths = HealthHelper.getHealths();
		for (Health h : healths) {
			if (h.getUuid().equals(uuid)) 
				return true;
		}

		return false;
	}
}
