package com.peak.tools.easyexcel.listener;

import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.converters.date.DateNumberConverter;
import com.alibaba.excel.enums.CellDataTypeEnum;
import com.alibaba.excel.exception.ExcelAnalysisStopException;
import com.alibaba.excel.exception.ExcelDataConvertException;
import com.alibaba.excel.metadata.CellExtra;
import com.alibaba.excel.metadata.data.DataFormatData;
import com.alibaba.excel.metadata.data.ReadCellData;
import com.alibaba.excel.read.listener.ReadListener;
import com.alibaba.excel.read.metadata.holder.ReadRowHolder;
import com.alibaba.excel.util.ListUtils;
import com.peak.prd.base.annotation.ExcelErrorMsg;
import com.peak.prd.base.annotation.ExcelImportMaxLength;
import com.peak.prd.base.annotation.ExcelImportNotBlank;
import com.peak.prd.base.annotation.ExcelImportNotNull;
import com.peak.prd.base.annotation.ExcelImportValidInt;
import com.peak.tools.easyexcel.model.BaseExcelModel;
import com.peak.tools.easyexcel.model.ExcelReadData;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;

/**
 * <p>easyexcel读取监听处理类</p>
 * @author ldc
 * @version 1.0
 * @date 2022-5-13 10:42:50
 *
 * @param <T>
 */
public class ExcelDateListener<T> implements ReadListener<T> {

	/**开始行数*/
	private int beginRowIndex;
	
	/**结束行数*/
	private int endRowIndex;
	
	/**数据的开始读取行数，行号从0开始*/
	private int firstRowIndex = 0;
	
	/**excel读取后实体数据对象*/
	private ExcelReadData<T> excelReadData;
	
	private List<CellExtra> extraMergeInfoList = new ArrayList<>();
	
	/**model class*/
	private Class<T> dataClass;
	
	
	public ExcelDateListener(Class<T> dataClass, int firstRowIndex, int pageNumber, int pageSize) {
		this.beginRowIndex = (pageNumber - 1) * pageSize;
		this.endRowIndex = pageNumber * pageSize;
		this.firstRowIndex = firstRowIndex;
		this.dataClass = dataClass;
		excelReadData = new ExcelReadData<T>();
	}
	
	@Override
	public void invoke(T data, AnalysisContext context) {
		//获取行号
    	ReadRowHolder readRowHolder = context.readRowHolder();
    	Integer rowIndex = readRowHolder.getRowIndex();

        //只要当前页的信息
    	if (rowIndex >= beginRowIndex + firstRowIndex && rowIndex < endRowIndex + firstRowIndex) {
    		
    		//空行不处理数据，ExcelReadData的连续为空行数中加1，后续逻辑在ExcelReadData中处理
        	if (isRowNullValue(data)) {
        		excelReadData.addSerialNullRows();
                return;
            }
        	
        	excelReadData.setSerialNullRows(1);
    		
    		Field[] fields = data.getClass().getDeclaredFields();
            for (Field field : fields) {
                //设置可访问
                field.setAccessible(true);
                //属性的值
                Object fieldValue = null;
                try {
                    fieldValue = field.get(data);

                    //是否包含不能为空的校验注解
                    ExcelImportNotNull excelImportNotNull = field.getAnnotation(ExcelImportNotNull.class);
                    
                    //包含不能为空的校验注解当前值还是null
                    if (excelImportNotNull != null) {
                    	if(Objects.isNull(fieldValue)) {
	                    	//错误列表放入数据
	                        ((BaseExcelModel) data).setExcelImportErrorMessage(excelImportNotNull.value());
	                    	excelReadData.addReadErrorData(data, rowIndex - firstRowIndex + 1);
	                    	return;
                    	}
                    	
                    	//正整数校验
                    	ExcelImportValidInt excelImportValidInt = field.getAnnotation(ExcelImportValidInt.class);
                    	if(excelImportValidInt != null) {
                    		try {
								Long.parseLong(fieldValue.toString());
								if(fieldValue.toString().indexOf(".") >= 0) {
									//错误列表放入数据
			                        ((BaseExcelModel) data).setExcelImportErrorMessage(excelImportValidInt.value());
			                    	excelReadData.addReadErrorData(data, rowIndex - firstRowIndex + 1);
			                    	return;
								}
							} catch (Exception e) {
								//错误列表放入数据
		                        ((BaseExcelModel) data).setExcelImportErrorMessage(excelImportValidInt.value());
		                    	excelReadData.addReadErrorData(data, rowIndex - firstRowIndex + 1);
		                    	return;
							}
                    	}
                    	
                    } else {
                    	//是否包含字符串不为空校验注解
                    	ExcelImportNotBlank excelImportNotBlank = field.getAnnotation(ExcelImportNotBlank.class);
                    	
                    	//是否包含最大值校验注解
                        ExcelImportMaxLength excelImportMaxLength = field.getAnnotation(ExcelImportMaxLength.class);
                         
                        //包含不能字符串不为空校验注解,并且字符串校验不通过
                        if(excelImportNotBlank != null && (Objects.isNull(fieldValue) || fieldValue.toString().trim().length() == 0)) {
                        	 
                        	//错误列表放入数据
                        	((BaseExcelModel) data).setExcelImportErrorMessage(excelImportNotBlank.value());
                          	excelReadData.addReadErrorData(data, rowIndex - firstRowIndex + 1);
                        	return;
                        } else if(excelImportMaxLength != null && fieldValue.toString().trim().length() > excelImportMaxLength.value()) {
                        	//包含是否包含最大值校验注解,并且字符串校验不通过,错误列表放入数据
                        	((BaseExcelModel) data).setExcelImportErrorMessage(excelImportMaxLength.errorMsg());
                          	excelReadData.addReadErrorData(data, rowIndex - firstRowIndex + 1);
                        	return;
                        }
                        
                    }
                    
                    
                } catch(Exception e) {
                	//反射出现异常不做处理,为了防止丢失数据finally放到成功列表中
                }
            }
            //校验通过放到成功列表中
            excelReadData.getDataList().add(data);
    	}
	}
	
	 /**
     * <p>在转换异常 获取其他异常下会调用本接口。抛出异常则停止读取。如果这里不抛出异常则 继续读取下一行。</p>
     *
     * @param exception
     * @param context
     * @throws Exception
     */
	@Override
	public void onException(Exception exception, AnalysisContext context) {
		if (exception instanceof ExcelDataConvertException) {
			try {
				ExcelDataConvertException excelDataConvertException = (ExcelDataConvertException)exception;
				int errorRowIndex = excelDataConvertException.getRowIndex();
				//错误的行数大于开始的行数
				if(errorRowIndex >= beginRowIndex + firstRowIndex && errorRowIndex < endRowIndex + firstRowIndex) {
					List<Object> errordata = ListUtils.newArrayList();

					Object object = context.readRowHolder().getCurrentRowAnalysisResult();;

					if(object != null) {
						Map<Integer, ReadCellData> datamap =  (Map<Integer, ReadCellData>) object;

						for (Entry<Integer, ReadCellData> entry : datamap.entrySet()) {
							ReadCellData readCellData = entry.getValue();
							CellDataTypeEnum type = readCellData.getType();

							DataFormatData dataFormatData = readCellData.getDataFormatData();
							String format = dataFormatData.getFormat();

							if(type == CellDataTypeEnum.BOOLEAN) {
								errordata.add(readCellData.getBooleanValue());
							} else if(type == CellDataTypeEnum.STRING || type == CellDataTypeEnum.DATE) {
								errordata.add(readCellData.getStringValue());
							} else if(type == CellDataTypeEnum.NUMBER) {
								//日期格式easyexcel中也解析成number
								if(format != null && format.equals("yyyy\\-mm\\-dd\\ hh:mm:ss")) {
									DateNumberConverter dateNumberConverter = new DateNumberConverter();
									Date date = dateNumberConverter.convertToJavaData(readCellData,
											excelDataConvertException.getExcelContentProperty(), context.readRowHolder().getGlobalConfiguration());
									errordata.add(date);
								} else {
									errordata.add(readCellData.getNumberValue());
								}
							} else {
								errordata.add("");
							}
						}

						//错误发生列
						int errorColumnIndex = excelDataConvertException.getColumnIndex().intValue();
						//默认错误信息
						String errorMsg = "第【"+ (errorColumnIndex + 1) + "】列的数据格式错误！！";

						try {
							//获得数据模型对象所有的属性
							Field[] fields = dataClass.getDeclaredFields();
							//错误发生列对应的属性
							Field field = fields[errorColumnIndex];

							ExcelErrorMsg excelErrorMsg = field.getAnnotation(ExcelErrorMsg.class);
							if(excelErrorMsg != null) {
								//设置注解值为错误信息
								errorMsg = excelErrorMsg.value();
							}

						} catch (Exception e) {}

						errordata.add(errorMsg);
						excelReadData.addReadErrorData(errordata, errorRowIndex - firstRowIndex + 1);

					}
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else if (exception instanceof ExcelAnalysisStopException) {
			throw new ExcelAnalysisStopException();
		}
	}

    /**
     * <p>这里会一行行的返回头</p>
     *
     * @param headMap
     * @param context
     */
    @Override
    public void invokeHead(Map<Integer, ReadCellData<?>> headMap, AnalysisContext context) {
//    	//获取行号
//    	ReadRowHolder readRowHolder = context.readRowHolder();
//    	Integer rowIndex = readRowHolder.getRowIndex();
    }

	/**
	*<p>当返回额外信息时，将调用当前方法</p>
	* @param		extra
	* @param		context
	*/
	@Override
	public void extra(CellExtra extra, AnalysisContext context) { 
		switch (extra.getType()) {
			case MERGE:
			if (extra.getRowIndex() >= firstRowIndex) {
				extraMergeInfoList.add(extra);
            }
			break;
		}
	}

    /**
     * <p>验证是否存在另一条数据。您可以通过返回false来停止读取</p>
     * @param context
     * @return
     */
	public boolean hasNext(AnalysisContext context) {
		//获取行号
    	ReadRowHolder readRowHolder = context.readRowHolder();
    	Integer rowIndex = readRowHolder.getRowIndex();

        return rowIndex >= endRowIndex + firstRowIndex ? false : true;
    }
	
	/**
	 * <p>读取结束后执行</p>
     * @param context
	 */
	@Override
	public void doAfterAllAnalysed(AnalysisContext context) {
		
	}
	
	public ExcelReadData<T> getExcelReadData() {
		return this.excelReadData;
	}
	
	public List<CellExtra> getExtraMergeInfoList() {
		return extraMergeInfoList;
	}

	/**
     * 判断整行单元格数据是否均为空
     */
	private boolean isRowNullValue(T data) {
		if (data instanceof String) {
			return Objects.isNull(data);
		}
		try {
			Field[] fields = data.getClass().getDeclaredFields();
			List<Boolean> lineNullList = new ArrayList<>();
			for (Field field : fields) {
				field.setAccessible(true);
				Object value = field.get(data);
				if (Objects.isNull(value)) {
					lineNullList.add(Boolean.TRUE);
				} else {
					lineNullList.add(Boolean.FALSE);
				}
			}
			return lineNullList.stream().allMatch(Boolean.TRUE::equals);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

}
