package com.peak.tools.easyexcel.util;

import java.io.File;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.annotation.ExcelProperty;
import com.alibaba.excel.enums.CellExtraTypeEnum;
import com.alibaba.excel.metadata.CellExtra;
import com.alibaba.excel.util.ListUtils;
import com.peak.common.util.AppPathUtil;
import com.peak.common.util.ConvertUtil;
import com.peak.common.util.DateUtil;
import com.peak.common.util.HttpUtil;
import com.peak.common.util.ListUtil;
import com.peak.tools.easyexcel.listener.ExcelDateListener;
import com.peak.tools.easyexcel.model.ExcelReadData;
import com.peak.tools.easyexcel.model.ExcelResultVO;
import com.peak.tools.easyexcel.model.ReadErrorData;
import com.peak.prd.base.model.ResponseStatusEnum;
import com.peak.prd.base.util.ExceptionUtil;

/**
 * 
 * <p>easyexcel读取处理类</p>
 * @author ldc
 * @version 1.0
 * @date 2022-5-9 15:29:52
 *
 */
public final class ExcelReadUtil {
	
	/**默认每页2000行数据*/
	private int pageSize = 2000;
	
	/**最大允许的导入错误2000行数据*/
	private static final int ERROR_MAXNUM = 2000;
	
	/**读取总数量*/
	private int readTotalNum = 0;

	private int firstDataRowIndex = 1;
	
	
	
	/**总错误列表*/
	private List<ReadErrorData> errorDataTotalList = new ArrayList<ReadErrorData>();
	
	
	/**
	 * <p>获得excel读取后实体数据对象</p>
	 * @param <T>				数据模型
	 * @param pathName			Excel文件路径+文件名
	 * @param modelClass		T.class
	 * @param firstDataRowIndex 数据的开始读取行数，行号从0开始
	 * @param pageNumber		当前页
	 * @return List<T>			返回excel读取后实体数据对象列表
	 * @author ldc
	 * @version 1.0 
	 * @date 2022-5-12 11:19:42
	 */
	public <T> List<T> getExcelReadData(String pathName, Class<T> modelClass, int firstDataRowIndex, int pageNumber) {
		return getExcelReadData(pathName, modelClass, firstDataRowIndex, pageNumber, this.pageSize);
	}
	
	/**
	 * <p>获得excel读取后实体数据对象</p>
	 * @param <T>				数据模型
	 * @param sheetNo			sheet页码，从0开始
	 * @param pathName			Excel文件路径+文件名
	 * @param modelClass		T.class
	 * @param firstDataRowIndex 数据的开始读取行数，行号从0开始
	 * @param pageNumber		当前页
	 * @return List<T>			返回excel读取后实体数据对象列表
	 * @author ldc
	 * @date   2023年7月27日 下午7:23:20
	 */
	public <T> List<T> getExcelReadData(int sheetNo, String pathName, Class<T> modelClass, int firstDataRowIndex, int pageNumber) {
		return getExcelReadData(sheetNo, pathName, modelClass, firstDataRowIndex, pageNumber, this.pageSize);
	}
	
	/**
	 * <p>获得excel读取后实体数据对象</p>
	 * @param <T>				数据模型
	 * @param pathName			Excel文件路径+文件名
	 * @param modelClass		T.class
	 * @param firstDataRowIndex 数据的开始读取行数，行号从0开始
	 * @param pageNumber		当前页
	 * @param pageSize			每页条数
	 * @return List<T>			返回excel读取后实体数据对象列表
	 * @author ldc
	 * @version 1.0 
	 * @date 2022-5-12 11:19:42
	 */
	public <T> List<T> getExcelReadData(String pathName, Class<T> modelClass, int firstDataRowIndex, int pageNumber, int pageSize) {
		return getExcelReadData(0, pathName, modelClass, firstDataRowIndex, pageNumber);
	}
	
	/**
	 * <p>获得excel读取后实体数据对象</p>
	 * @param <T>				数据模型
	 * @param sheetNo			sheet页码，从0开始
	 * @param pathName			Excel文件路径+文件名
	 * @param modelClass		T.class
	 * @param firstDataRowIndex 数据的开始读取行数，行号从0开始
	 * @param pageNumber		当前页
	 * @param pageSize			每页条数
	 * @return List<T>			返回excel读取后实体数据对象列表
	 * @author ldc
	 * @version 1.0 
	 * @date   2023年7月27日 下午7:24:43
	 */
	public <T> List<T> getExcelReadData(int sheetNo, String pathName, Class<T> modelClass, int firstDataRowIndex, int pageNumber, int pageSize) {
		
		this.pageSize = pageSize;

		this.firstDataRowIndex = firstDataRowIndex;
		
		//读取Excel,获得excel读取后实体数据对象ExcelReadData
		ExcelReadData<T> excelReadData = ExcelReadUtil.executeExcelReadData(sheetNo, pathName, modelClass, firstDataRowIndex, pageNumber, this.pageSize);
		
		//本次读取成功数量
		int readDataNum = excelReadData.getDataList().size();
		
		//本次读取失败数量
		int readErrorNum = excelReadData.getErrorList().size();
		
		//重新设置总数量
		readTotalNum += readDataNum + readErrorNum;
		
		//本次读取错误列表放入总错误列表
		if(readErrorNum > 0) {
			this.errorDataTotalList.addAll(excelReadData.getErrorList());
			
			//错误的数量大于一页的数量时抛出异常结束读取
			if(errorDataTotalList.size() >= ERROR_MAXNUM) {
				//错误数据的excel最后一行加入“错误行数已经超过2000行，不再继续导入，请您重新整理后再次导入”提示信息数据
				addReadMaxErrorData();
				
				ExceptionUtil.handle(ResponseStatusEnum.EXCEL_READ_ERRORNUMOVERSTEP);
			}
		}
		
		return excelReadData.getDataList();
	}
	
	private static <T> ExcelReadData<T> executeExcelReadData(int sheetNo, String pathName, Class<T> modelClass, int firstDataRowIndex, int pageNumber, int pageSize) {
		
		//实例化ExcelDateListener
		ExcelDateListener<T> excelDateListener = new ExcelDateListener<T>(modelClass, firstDataRowIndex, pageNumber, pageSize);
	   
		//这里需要指定读用哪个class去读，然后读取第一个sheet 文件流会自动关闭
	    EasyExcel.read(pathName, modelClass, excelDateListener).sheet(sheetNo).doRead();
	    
	    //获得excel读取后实体数据对象
	    return excelDateListener.getExcelReadData();
	    
	}
	

	/**
	 * <p>获得excel读取后包括合并单元格的实体数据对象</p>
	 * @param <T>
	 * @param pathName
	 * @param modelClass
	 * @param firstDataRowIndex
	 * @param pageNumber
	 * @return
	 * @author ldc
	 * @date   2023年5月29日 下午3:30:33
	 */
	public <T> List<T> getExcelMergeReadData(String pathName, Class<T> modelClass, int firstDataRowIndex, int pageNumber) {
		return getExcelMergeReadData(0, pathName, modelClass, firstDataRowIndex, pageNumber, this.pageSize);
	}
	
	/**
	 * 
	 * <p>获得excel读取后包括合并单元格的实体数据对象</p>
	 * @param <T>
	 * @param sheetNo			sheet页码，从0开始
	 * @param pathName
	 * @param modelClass
	 * @param firstDataRowIndex
	 * @param pageNumber
	 * @return
	 * @author ldc
	 * @date   2023年7月27日 下午7:29:20
	 */
	public <T> List<T> getExcelMergeReadData(int sheetNo, String pathName, Class<T> modelClass, int firstDataRowIndex, int pageNumber) {
		return getExcelMergeReadData(sheetNo, pathName, modelClass, firstDataRowIndex, pageNumber, this.pageSize);
	}
	
	/**
	 * <p>获得excel读取后实体数据对象</p>
	 * @param <T>				数据模型
	 * @param pathName			Excel文件路径+文件名
	 * @param modelClass		T.class
	 * @param firstDataRowIndex 数据的开始读取行数，行号从0开始
	 * @param pageNumber		当前页
	 * @param pageSize			每页条数
	 * @return List<T>			返回excel读取后实体数据对象列表
	 * @author ldc
	 * @version 1.0 
	 * @date 2023-5-29 15:33:42
	 */
	public <T> List<T> getExcelMergeReadData(String pathName, Class<T> modelClass, int firstDataRowIndex, int pageNumber, int pageSize) {
		return this.getExcelMergeReadData(0, pathName, modelClass, firstDataRowIndex, pageNumber, pageSize);
		
	}
	
	/**
	 * <p>获得excel读取后实体数据对象</p>
	 * @param <T>				数据模型
	 * @param sheetNo			sheet页码，从0开始
	 * @param pathName			Excel文件路径+文件名
	 * @param modelClass		T.class
	 * @param firstDataRowIndex 数据的开始读取行数，行号从0开始
	 * @param pageNumber		当前页
	 * @param pageSize			每页条数
	 * @return List<T>			返回excel读取后实体数据对象列表
	 * @author ldc
	 * @date   2023年7月27日 下午7:29:58
	 */
	public <T> List<T> getExcelMergeReadData(int sheetNo, String pathName, Class<T> modelClass, int firstDataRowIndex, int pageNumber, int pageSize) {
		this.pageSize = pageSize;

		this.firstDataRowIndex = firstDataRowIndex;
		
		//读取Excel,获得excel读取后实体数据对象ExcelReadData
		ExcelDateListener<T> excelDateListener = ExcelReadUtil.executeExcelMergeReadData(sheetNo, pathName, modelClass, firstDataRowIndex, pageNumber, this.pageSize);

		
		ExcelReadData<T> excelReadData = excelDateListener.getExcelReadData();
		
		if(ListUtil.notNullorEmpty(excelReadData.getDataList())) {
			explainMergeData(excelReadData.getDataList(), excelDateListener.getExtraMergeInfoList(), firstDataRowIndex);
		}
		
		//本次读取成功数量
		int readDataNum = excelReadData.getDataList().size();
		
		//本次读取失败数量
		int readErrorNum = excelReadData.getErrorList().size();
		
		//重新设置总数量
		readTotalNum += readDataNum + readErrorNum;
		
		//本次读取错误列表放入总错误列表
		if(readErrorNum > 0) {
			this.errorDataTotalList.addAll(excelReadData.getErrorList());
			
			//错误的数量大于一页的数量时抛出异常结束读取
			if(errorDataTotalList.size() >= ERROR_MAXNUM) {
				//错误数据的excel最后一行加入“错误行数已经超过2000行，不再继续导入，请您重新整理后再次导入”提示信息数据
				addReadMaxErrorData();
				
				ExceptionUtil.handle(ResponseStatusEnum.EXCEL_READ_ERRORNUMOVERSTEP);
			}
		}
		
		return excelReadData.getDataList();
	}
	
	private static <T> ExcelDateListener<T> executeExcelMergeReadData(int sheetNo, String pathName, Class<T> modelClass, int firstDataRowIndex, int pageNumber, int pageSize) {
		
		//实例化ExcelDateListener
		ExcelDateListener<T> excelDateListener = new ExcelDateListener<T>(modelClass, firstDataRowIndex, pageNumber, pageSize);
	   
		//这里需要指定读用哪个class去读，然后读取第一个sheet 文件流会自动关闭
	    EasyExcel.read(pathName, modelClass, excelDateListener).extraRead(CellExtraTypeEnum.MERGE).sheet(sheetNo).doRead();
	    
	    //获得excel读取后实体数据对象
	    return excelDateListener;
	    
	}
	
	/**
     * 处理合并单元格
     *
     * @param data               解析数据
     * @param extraMergeInfoList 合并单元格信息
     * @param headRowNumber      起始行
     * @return 填充好的解析数据
     */
	private static <T> List<T> explainMergeData(List<T> data, List<CellExtra> extraMergeInfoList, Integer headRowNumber) {
        //循环所有合并单元格信息
        extraMergeInfoList.forEach(cellExtra -> {
            int firstRowIndex = cellExtra.getFirstRowIndex() - headRowNumber;
            int lastRowIndex = cellExtra.getLastRowIndex() - headRowNumber;
            int firstColumnIndex = cellExtra.getFirstColumnIndex();
            int lastColumnIndex = cellExtra.getLastColumnIndex();
            //获取初始值
            Object initValue = getInitValueFromList(firstRowIndex, firstColumnIndex, data);
            //设置值
            for (int i = firstRowIndex; i <= lastRowIndex; i++) {
                for (int j = firstColumnIndex; j <= lastColumnIndex; j++) {
                    setInitValueToList(initValue, i, j, data);
                }
            }
        });
        return data;
    }
	 
	 /**
     * 设置合并单元格的值
     *
     * @param filedValue  值
     * @param rowIndex    行
     * @param columnIndex 列
     * @param data        解析数据
     */
    private static <T> void setInitValueToList(Object filedValue, Integer rowIndex, Integer columnIndex, List<T> data) {
        T object = data.get(rowIndex);

        for (Field field : object.getClass().getDeclaredFields()) {
            //提升反射性能，关闭安全检查
            field.setAccessible(true);
            ExcelProperty annotation = field.getAnnotation(ExcelProperty.class);
            if (annotation != null) {
                if (annotation.index() == columnIndex) {
                    try {
                        field.set(object, filedValue);
                        break;
                    } catch (IllegalAccessException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }


    /**
     * 获取合并单元格的初始值
     * rowIndex对应list的索引
     * columnIndex对应实体内的字段
     *
     * @param firstRowIndex    起始行
     * @param firstColumnIndex 起始列
     * @param data             列数据
     * @return 初始值
     */
    private static <T> Object getInitValueFromList(Integer firstRowIndex, Integer firstColumnIndex, List<T> data) {
        Object filedValue = null;
        T object = data.get(firstRowIndex);
        for (Field field : object.getClass().getDeclaredFields()) {
            //提升反射性能，关闭安全检查
            field.setAccessible(true);
            ExcelProperty annotation = field.getAnnotation(ExcelProperty.class);
            if (annotation != null) {
                if (annotation.index() == firstColumnIndex) {
                    try {
                        filedValue = field.get(object);
                        break;
                    } catch (IllegalAccessException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return filedValue;
    }
	
	/**
	 * <p>增加导入错误信息对象</p>
	 * @param errorData			错误对象列表
	 * @param pageNumber		当前页
	 * @param pageSize			每页数量，没有指定的话使用
	 * @param rowIndex			错误发生行索引
	 * @author ldc
	 * @version 1.0
	 * @date 2022-5-12 17:12:01
	 */
	public <T> void addReadErrorData(T errorData, int pageNumber, int rowIndex) {
		rowIndex = (pageNumber - 1) * this.pageSize + rowIndex;
		ReadErrorData readErrorData = new ReadErrorData(errorData, rowIndex);
		this.errorDataTotalList.add(readErrorData);
		
		//错误的数量大于2000的数量时抛出异常结束读取
		if(errorDataTotalList.size() >= ERROR_MAXNUM) {
			//错误数据的excel最后一行加入“错误行数已经超过2000行，不再继续导入，请您重新整理后再次导入”提示信息数据
			addReadMaxErrorData();
			
			//抛出异常
			ExceptionUtil.handle(ResponseStatusEnum.EXCEL_READ_ERRORNUMOVERSTEP);
		}
	}
	
	/**
	 * <p>错误数据的excel最后一行加入“错误行数已经超过2000行，不再继续导入，请您重新整理后再次导入”提示信息数据</p>
	 * @author ldc
	 * @version 1.0
	 * @date 2022-5-16 15:08:40
	 */
	private void addReadMaxErrorData() {
		//错误数据的excel最后一行加入“错误行数已经超过2000行，不再继续导入，请您重新整理后再次导入”提示信息
		List<Object> errordata = ListUtils.newArrayList();
		errordata.add("错误行数已经超过2000行，不再继续导入，请您重新整理后再次导入");
		ReadErrorData readMaxErrorData = new ReadErrorData(errordata, Integer.MAX_VALUE);
		this.errorDataTotalList.add(readMaxErrorData);
		
		//抛出异常
		ExceptionUtil.handle(ResponseStatusEnum.EXCEL_READ_ERRORNUMOVERSTEP);
	}

 	/**
	 * <p>获得成功读取数量</p>
	 * @return
	 * @author ldc
	 * @version 1.0
	 * @date 2022-5-12 14:26:55
	 */
	public int getReadDataNum() {
		return readTotalNum - errorDataTotalList.size();
	}
	
	/**
	 * <p>获得错误读取数量</p>
	 * @return
	 * @author ldc
	 * @version 1.0
	 * @data 2022-5-12 14:27:00
	 */
	public int getReadErrorNum() {
		return readTotalNum - getReadDataNum();
	}

	/**
	 * <p>获得读取总数量</p>
	 * @return
	 * @author ldc
	 * @version 2022-5-12 14:22:35
	 */
	public int getReadTotalNum() {
		return readTotalNum;
	}

	/**
	 * <p>获得排序好的错误总列表数据</p>
	 * @return
	 * @author ldc
	 * @version 1.0
	 * @date 2022-5-12 18:24:45
	 */
	public List<Object> getErrorDatalList() {
		//排序ReadErrorData中的index字段
		Collections.sort(errorDataTotalList);
		
		//总错误列表转换成ReadErrorData中的errorDate对象列表返回
		List<Object> errorList = ConvertUtil.convert(this.errorDataTotalList, e -> e.getErrorDate());
		return errorList == null ? new ArrayList<Object>() : errorList;
	}
	
	/**
	 * <p>获得ExcelResultVO,在调用此方法之前已经生成了错误的excel文件使用</p>
	 * @param pathName			错误的yyyy/MM/dd文件路径/文件名称,比如：2022/05/13/test.xlsx
	 * @return
	 * @author ldc
	 * @version 1.0
	 * @date 2022-5-13 15:11:28
	 * @deprecated
	 */
	private ExcelResultVO getExcelResultVO(String pathName) {
		ExcelResultVO excelResultVO = new ExcelResultVO();
		excelResultVO.setReadTotalNum(getReadTotalNum());
		excelResultVO.setReadSuccessNum(getReadDataNum());
		excelResultVO.setReadErrorNum(getReadErrorNum());
		
		if(getReadErrorNum() > 0) {
			String errorExcelFilepath = HttpUtil.formatURL("errorexcelpath/" + pathName);
			excelResultVO.setErrorExcelFilepath(errorExcelFilepath);
		}
		
		return excelResultVO;
	}
	
	/**
	 * <p>获得ExcelResultVO,在调用此方法之前没用生成错误的excel文件使用，该方法会自动生成错误的excel文件</p>
	 * @param <T>
	 * @param modelclass				按照数据模型.class
	 * @return
	 * @author ldc
	 * @version 1.0
	 * @date 2022-5-13 15:02:32
	 */
	public <T> ExcelResultVO getExcelResultVO(Class<T> modelclass) {
		
		ExcelResultVO excelResultVO = new ExcelResultVO();
		excelResultVO.setReadTotalNum(getReadTotalNum());
		excelResultVO.setReadSuccessNum(getReadDataNum());
		excelResultVO.setReadErrorNum(getReadErrorNum());
		
		if(getReadErrorNum() > 0) {
			
			Date curdate = new Date();
			int year = DateUtil.getYear(curdate);
			int month = DateUtil.getMonth(curdate);
			int day = DateUtil.getDay(curdate);
			
			//导出的错误excel文件路径
			String filename = UUID.randomUUID().toString() + ".xlsx";
			String filepath = "errorexcelpath/" + year + "/" + month + "/" + day + "/";

			ExcelWriteUtil excelWriteUtil = null;
			
			try {
				//获得本次读取错误的全部数据
				List<Object> errorList = this.getErrorDatalList();
				if(errorList != null && errorList.size() > 0) {
					
					//导出的错误excel文件全路径,包括共享资源目录的本地路径
					//2023-12-05 ldc 在产品111版本将错误文件的路径改成临时路径
					String localPath = AppPathUtil.getTempLocalPath();
					String fullpath = localPath + "/" + filepath;
					if(localPath.endsWith("/")) {
						fullpath = localPath + filepath;
					} 
					
					//String fullpath = "D://" + filepath;
					
					//文件夹不存在自动生成
					File destFilePath = new File(fullpath);
					if (!destFilePath.exists()) {
						destFilePath.mkdirs();
					}
					
					//写入Excel,这里用于写入导入的错误信息
					excelWriteUtil = new ExcelWriteUtil(fullpath + filename, modelclass);
					if(this.firstDataRowIndex == 2) {
						//模版从第二行开始才合并第一次行为不正确的数据列表
						excelWriteUtil.excelWrite("错误数据", "不正确的数据列表", errorList, true);
					} else {
						//模版从第二行开始才合并第一次行为不正确的数据列表
						excelWriteUtil.excelWrite("错误数据", errorList, true);
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
			} finally {
				//写入结束关闭流
				if(excelWriteUtil != null) {
					excelWriteUtil.finishWrite();
				}
			}
			
			String errorExcelFilepath = HttpUtil.formatURL(filepath + filename);
			excelResultVO.setErrorExcelFilepath(errorExcelFilepath);
		}
		
		return excelResultVO;
	}
}
