package com.peak.tools.easyexcel.util;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.annotation.ExcelProperty;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.excel.write.style.column.SimpleColumnWidthStyleStrategy;
import com.peak.common.util.*;
import com.peak.prd.base.model.ResponseStatusEnum;
import com.peak.prd.base.service.IOssService;
import com.peak.prd.base.util.ExceptionUtil;
import com.peak.prd.config.AppCommonConfig;
import com.peak.tools.easyexcel.model.BaseExcelModel;
import com.peak.tools.easyexcel.model.ExcelModelData;
import com.peak.tools.easyexcel.data.ExcelWriteBusinessDataUtil;
import com.peak.tools.easyexcel.model.ExportExcelVO;

/**
 *
 * <p>easyexcel写入处理类</p>
 * @author ldc
 * @version 1.0
 * @date 2022-5-9 15:29:18
 *
 */
public final class ExcelWriteUtil {

	/**默认每页2000行数据*/
	private int pageSize = 2000;

	/**写文件流*/
	private ExcelWriter excelWriter = null;

	/**当前写文件流是否关闭，默认没打开就是关闭*/
	private boolean isFinishWrite = true;

	private String excelWritePath;

	//oss上传目录类型 1：resource目录 2：temp目录 ，默认是temp目录
	private int ossUploadFileType = OSS_UPLOAD_TEMP_TYPE;

	public static final int OSS_UPLOAD_RESOURCE_TYPE = 1;
	public static final int OSS_UPLOAD_TEMP_TYPE = 2;


	/**
	 * @param pathName			文件路径+文件名
	 * @param modelClass		数据模型.class
	 */
	public ExcelWriteUtil(String pathName, Class<?> modelClass) {
		changeExcelHead(modelClass);
		excelWriter = EasyExcel.write(pathName, modelClass).build();
		this.excelWritePath = pathName;
		//设置的目录中不包含临时目录，就设置oss上传目录类型是resource目录
		if(excelWritePath.indexOf(AppPathUtil.getTempLocalPath()) == -1) {
			this.ossUploadFileType = OSS_UPLOAD_RESOURCE_TYPE;
		}
	}

	/**
	 * @param modelClass		数据模型.class
	 */
	public ExcelWriteUtil(Class<?> modelClass) {
		changeExcelHead(modelClass);
		excelWriter = EasyExcel.write(getPathName(), modelClass).build();
	}

	/**
	 * @param pathName			文件路径+文件名
	 */
	public ExcelWriteUtil(String pathName) {
		excelWriter = EasyExcel.write(pathName).build();
		this.excelWritePath = pathName;
		//设置的目录中不包含临时目录，就设置oss上传目录类型是resource目录
		if(excelWritePath.indexOf(AppPathUtil.getTempLocalPath()) == -1) {
			this.ossUploadFileType = OSS_UPLOAD_RESOURCE_TYPE;
		}
	}


	public ExcelWriteUtil() {
		excelWriter = EasyExcel.write(getPathName()).build();
	}

	/**
	 * <p>替换modelClass中的原列名为新列名</p>
	 * @param modelClass
	 * @author ldc
	 * @date   2023年12月18日 下午5:43:22
	 */
	private void changeExcelHead(Class<?> modelClass) {
		try{
			//获得替换的列名map
			Map<String, String> excelWriteColumnsReplaceMap = this.getExcelWriteColumnsReplaceMap();
			//替换的列名不存在直接返回
			if(excelWriteColumnsReplaceMap == null || excelWriteColumnsReplaceMap.size() == 0) return;

			Field[] fields = modelClass.getDeclaredFields();
			for(Field field : fields) {
				field.setAccessible(true);
				ExcelProperty property = field.getAnnotation(ExcelProperty.class);
				if(property != null) {
					InvocationHandler invocationHandler = Proxy.getInvocationHandler(property);
					Field memberValues = invocationHandler.getClass().getDeclaredField("memberValues");
					memberValues.setAccessible(true);
					Map<String, Object> values = (Map<String, Object>) memberValues.get(invocationHandler);
					String value = property.value()[0];
					String replaceValue = excelWriteColumnsReplaceMap.get(value);
					if(replaceValue != null && replaceValue.trim().length() > 0) {
						value = replaceValue;
					}
					values.put("value", new String[]{value});
				}
			}
		} catch (Exception e){
			e.printStackTrace();
		}
	}

	/**
	 * <p>获得替换的列名map</p>
	 * @return
	 * @author ldc
	 * @throws UnsupportedEncodingException
	 * @date   2023年12月18日 下午5:54:45
	 */
	private Map<String, String> getExcelWriteColumnsReplaceMap() throws UnsupportedEncodingException {
		//获得excel导出列名替换
		AppCommonConfig config = SpringUtil.getBean(AppCommonConfig.class);
		String ewcrvalue = config.getExcelWriteColumnsReplace();

		//没有需要替换的列名直接返回null
		if(StringUtil.isNullorEmptyStr(ewcrvalue)) {
			return null;
		}

		String utf8EwcrValue = new String(ewcrvalue.getBytes("ISO-8859-1"), "UTF-8");

		//替换的列名拆分成多个，不存在也直接返回null
		String[] excelWriteColumnsReplaces = utf8EwcrValue.split(",");
		if(excelWriteColumnsReplaces == null || excelWriteColumnsReplaces.length == 0) {
			return null;
		}

		Map<String, String> excelWriteColumnsReplaceMap = new HashMap<>();
		for(String ewcr : excelWriteColumnsReplaces) {
			String[] key_value = ewcr.split(":");
			if(key_value == null || key_value.length != 2) {
				continue;
			}
			excelWriteColumnsReplaceMap.put(key_value[0], key_value[1]);
		}

		return excelWriteColumnsReplaceMap;
	}


	/**
	 * <p>data列表数据写入到Excel中,每次写入数据不能为空，并且数据最多2000条,不符合的data会导出失败</p>
	 * @param sheetName			sheet名
	 * @param data				列表数据
	 * param excludeFieldNames	不导出的属性数组
	 * @author ldc
	 * @version 1.0
	 * @date 2023-7-4 14:29:25
	 */
	public void excelWrite(String sheetName, List<?> data, String... excludeFieldNames) {
		excelWrite(sheetName, data, false, excludeFieldNames);
	}

	/**
	 * <p>data列表数据写入到Excel中,每次写入数据不能为空，并且数据最多2000条,不符合的data会导出失败</p>
	 * @param sheetName			sheet名
	 * @param data				列表数据
	 * @author ldc
	 * @version 1.0
	 * @date 2022-5-10 14:29:25
	 */
	public void excelWrite(String sheetName, List<?> data) {
		excelWrite(sheetName, data, false);
	}

	/**
	 * <p>导出动态列表格</p>
	 * @param sheetName
	 * @param heads				表头列表
	 * @param dataList			数据列表
	 * @author ldc
	 * @date   2024年5月4日 上午11:30:54
	 */
	public void excelWrite(String sheetName, List<String> heads, List<List<Object>> dataList) {

		//每次写入的数据不能为空,不符合的data会抛出异常
		if(dataList == null) ExceptionUtil.handle(ResponseStatusEnum.EXCEL_WRITER_ERROR_DATANULL);

		//每次写入的数据最多2000条,不符合的data会抛出异常
		if(dataList.size() > this.pageSize) ExceptionUtil.handle(ResponseStatusEnum.EXCEL_WRITER_ERROR_DATAOVERSTEP);

		List<List<String>> hs = new ArrayList<>();
		for (String s : heads) {
			hs.add(Arrays.asList(s));
		}

		//执行了写入说明打开了文件流，文件流设置能没关闭
		isFinishWrite = false;
		WriteSheet writeSheet = EasyExcel.writerSheet(sheetName).registerWriteHandler(
				new SimpleColumnWidthStyleStrategy(16)).head(hs).build();


		excelWriter.write(dataList, writeSheet);
	}

	/**
	 * <p>data列表数据写入到Excel中,每次写入数据不能为空，并且数据最多2000条,不符合的data会导出失败</p>
	 * @param sheetName			sheet名
	 * @param data				列表数据
	 * @param isShowErrorMsg	是否写入错误信息 isShowErrorMsg=true写入，isShowErrorMsg=false不写入
	 * @author ldc
	 * @version 1.0
	 * @date 2022-5-10 14:29:25
	 */
	public void excelWrite(String sheetName, List<?> data, boolean isShowErrorMsg) {

		//每次写入的数据不能为空,不符合的data会抛出异常
		if(data == null) ExceptionUtil.handle(ResponseStatusEnum.EXCEL_WRITER_ERROR_DATANULL);

		//每次写入的数据最多2000条,不符合的data会抛出异常
		if(data.size() > this.pageSize) ExceptionUtil.handle(ResponseStatusEnum.EXCEL_WRITER_ERROR_DATAOVERSTEP);

		//执行了写入说明打开了文件流，文件流设置能没关闭
		isFinishWrite = false;
		WriteSheet writeSheet;
		if(isShowErrorMsg) {
			writeSheet = EasyExcel.writerSheet(sheetName).build();
		} else {
			Set<String> excludeField = new HashSet<String>();
			excludeField.add("excelImportErrorMessage");
			writeSheet = EasyExcel.writerSheet(sheetName).excludeColumnFiledNames(excludeField).build();
		}
		excelWriter.write(data, writeSheet);
	}

	/**
	 * <p>data列表数据写入到Excel中,每次写入数据不能为空，并且数据最多2000条,不符合的data会导出失败</p>
	 * @param sheetName         sheet名
	 * @param headTitle			表头名
	 * @param data				列表数据
	 * @param isShowErrorMsg	是否写入错误信息 isShowErrorMsg=true写入，isShowErrorMsg=false不写入
	 * @author ldc
	 * @date 2025/3/31 16:20
	 */
	public void excelWrite(String sheetName,String headTitle, List<?> data, boolean isShowErrorMsg) {

		//每次写入的数据不能为空,不符合的data会抛出异常
		if(data == null) ExceptionUtil.handle(ResponseStatusEnum.EXCEL_WRITER_ERROR_DATANULL);

		//每次写入的数据最多2000条,不符合的data会抛出异常
		if(data.size() > this.pageSize) ExceptionUtil.handle(ResponseStatusEnum.EXCEL_WRITER_ERROR_DATAOVERSTEP);

		//反射获得所有的model所有的属性
		Field[] modelfidlds = data.get(0).getClass().getDeclaredFields();
		List<List<String>> headList = new ArrayList<>();

		//遍历属性，重新设置表头
		for(Field modelfidld : modelfidlds) {
			ExcelProperty excelProperty = modelfidld.getAnnotation(ExcelProperty.class);
			String fieldshowname = "";
			if(excelProperty != null) {
				fieldshowname = excelProperty.value()[0];
			}
			headList.add(Arrays.asList(headTitle, fieldshowname));
		}

		headList.add(Arrays.asList(headTitle, "导入失败错误信息"));


		//执行了写入说明打开了文件流，文件流设置能没关闭
		isFinishWrite = false;
		WriteSheet writeSheet;
		if(isShowErrorMsg) {
			Set<String> excludeField = new HashSet<String>();
			excludeField.add("excelImportErrorMessage");
			writeSheet = EasyExcel.writerSheet(sheetName).head(headList).build();

		} else {
			Set<String> excludeField = new HashSet<String>();
			excludeField.add("excelImportErrorMessage");
			writeSheet = EasyExcel.writerSheet(sheetName).head(headList).
					excludeColumnFiledNames(excludeField).build();
		}
		excelWriter.write(data, writeSheet);
	}

	/**
	 * <p>根据新的model生成新的sheetName页，用于多个不同的sheet页使用</p>
	 * @param sheetName			sheetName页名称
	 * @param modelClass		model
	 * @param data			    数据列表
	 * @author ldc
	 * @date 2025/2/6 14:04
	*/
	public <Model> void excelWrite(String sheetName, Class<Model> modelClass, List<Model> data) {
		this.excelWrite(sheetName, modelClass, data, false);
	}

	/**
	 * <p>根据新的model生成新的sheetName页，用于多个不同的sheet页使用</p>
	 * @param sheetName			sheetName页名称
	 * @param modelClass		model
	 * @param data				数据列表
	 * @param isShowErrorMsg 	是否写入错误信息 isShowErrorMsg=true写入，isShowErrorMsg=false不写入
	 * @author ldc
	 * @date 2025/2/6 14:18
	*/
	public <Model> void excelWrite(String sheetName, Class<Model> modelClass, List<Model> data, boolean isShowErrorMsg) {

		//每次写入的数据不能为空,不符合的data会抛出异常
		if(data == null) ExceptionUtil.handle(ResponseStatusEnum.EXCEL_WRITER_ERROR_DATANULL);

		//每次写入的数据最多2000条,不符合的data会抛出异常
		if(data.size() > this.pageSize) ExceptionUtil.handle(ResponseStatusEnum.EXCEL_WRITER_ERROR_DATAOVERSTEP);

		//执行了写入说明打开了文件流，文件流设置能没关闭
		isFinishWrite = false;
		WriteSheet writeSheet;
		if(isShowErrorMsg) {
			writeSheet = EasyExcel.writerSheet(sheetName).head(modelClass).build();
		} else {
			Set<String> excludeField = new HashSet<String>();
			excludeField.add("excelImportErrorMessage");
			writeSheet = EasyExcel.writerSheet(sheetName).excludeColumnFiledNames(excludeField).head(modelClass).build();
		}
		excelWriter.write(data, writeSheet);
	}

	/**
	 * <p>data列表数据写入到Excel中,每次写入数据不能为空，并且数据最多2000条,不符合的data会导出失败</p>
	 * @param sheetName			sheet名
	 * @param data				列表数据
	 * @param isShowErrorMsg	是否写入错误信息 isShowErrorMsg=true写入，isShowErrorMsg=false不写入
	 * @param excludeFieldNames	不导出的属性数组
	 * @author ldc
	 * @version 1.0
	 * @date 2023-7-4 14:29:25
	 */
	public void excelWrite(String sheetName, List<?> data, boolean isShowErrorMsg, String... excludeFieldNames) {

		//每次写入的数据不能为空,不符合的data会抛出异常
		if(data == null) ExceptionUtil.handle(ResponseStatusEnum.EXCEL_WRITER_ERROR_DATANULL);

		//每次写入的数据最多2000条,不符合的data会抛出异常
		if(data.size() > this.pageSize) ExceptionUtil.handle(ResponseStatusEnum.EXCEL_WRITER_ERROR_DATAOVERSTEP);

		//执行了写入说明打开了文件流，文件流设置能没关闭
		isFinishWrite = false;
		WriteSheet writeSheet;
		Set<String> excludeField = new HashSet<String>();
		if(excludeFieldNames != null && excludeFieldNames.length > 0) {
			for(String excludeFieldName : excludeFieldNames) {
				excludeField.add(excludeFieldName);
			}
		}
		if(isShowErrorMsg) {
			writeSheet = EasyExcel.writerSheet(sheetName).excludeColumnFiledNames(excludeField).build();
		} else {
			excludeField.add("excelImportErrorMessage");
			writeSheet = EasyExcel.writerSheet(sheetName).excludeColumnFiledNames(excludeField).build();
		}
		excelWriter.write(data, writeSheet);
	}

	/**
	 * <p>关闭写文件流</p>
	 * @author ldc
	 * @version 1.0
	 * @date 2022-5-12 11:58:19
	 */
	public void finishWrite() {
		finishWriteNotUploadOss();

		//2023-11-30 ldc 上传导出文件到oss
		AppCommonConfig config = SpringUtil.getBean(AppCommonConfig.class);

		//使用oss存储
		if (config.getOssEnable()) {
			//上传文件到OSS
			IOssService ossService = (IOssService) SpringUtil.getBean(config.getOssServiceName());

			//如果ossUploadFileType目录是Resource目录，就使用Resource上传，否则使用temp目录上传
			if(this.ossUploadFileType == 1) { //oss上传目录类型是resource模式上传
				ossService.uploadFile(AppPathUtil.getResourceLocalPath(),
						this.excelWritePath.replace(AppPathUtil.getResourceLocalPath(), ""));
			} else {
				ossService.uploadFile(AppPathUtil.getTempLocalPath(),
						this.excelWritePath.replace(AppPathUtil.getTempLocalPath(), ""));
			}
		}
	}

	/**
	 * <p>关闭写文件流不上传到oss</p>
	 * @author ldc
	 * @date   2024年2月28日 下午3:36:51
	 */
	public void finishWriteNotUploadOss() {
		isFinishWrite = true; //设置写文件流关闭
		if(excelWriter != null) {
			excelWriter.finish(); //关闭流
		}
	}

	/**
	 * <p>文件流关闭状态</p>
	 * @return  返回值=false流没关闭状态，返回值=true流关闭状态
	 * @author  ldc
	 * @version 1.0
	 * @date 2022-5-12 11:57:39
	 */
	public boolean isFinishWrite() {
		return isFinishWrite;
	}

	/**
	 * <p>获得导出的excel文件路径，相对resource目录全路径，示例：excelwritepath/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.xlsx</p>
	 * @return
	 * @author ldc
	 * @version 1.0
	 * @date 2022-5-16 15:34:45
	 */
	public String getExcelWritePath() {
		return this.excelWritePath;
	}

	/**
	 * <p>获得导出的excel文件全路径</p>
	 * @return
	 * @author ldc
	 * @date   2023年9月15日 下午1:30:58
	 */
	public String getFullExcelWritePath() {
		return CommonUtil.formatByEndSlash(AppPathUtil.getTempLocalPath()) + this.excelWritePath;
	}

	private String getPathName() {

		Date curdate = new Date();
		int year = DateUtil.getYear(curdate);
		int month = DateUtil.getMonth(curdate);
		int day = DateUtil.getDay(curdate);

		//导出的错误excel文件路径
		String filename = UUID.randomUUID().toString() + ".xlsx";
		String excelWritePath = "excelwritepath/" + year + "/" + month + "/" + day + "/";

		this.excelWritePath = excelWritePath + filename;

		//导出的错误excel文件全路径,包括共享资源目录的本地路径
		String localPath = AppPathUtil.getTempLocalPath();
		String fullpath = localPath + "/" + excelWritePath;
		if(localPath.endsWith("/")) {
			fullpath = localPath + excelWritePath;
		}
		//String fullpath = "D://" + excelWritePath;

		//文件夹不存在自动生成
		File destFilePath = new File(fullpath);
		if (!destFilePath.exists()) {
			destFilePath.mkdirs();
		}

		return fullpath + filename;
	}

	public ExportExcelVO exportList(
			ExcelWriteBusinessDataUtil businessDataUtil,
			Class<? extends BaseExcelModel> excelModel,
			String sheetName, Long businessId) {
		//1、实例化ExcelWriteUtil工具类，需要将导出模型类的class放到构造方法中
		ExcelWriteUtil excelWriteUtil = new ExcelWriteUtil(excelModel);

//		2、创建读取数据循环中的变量，行号、页数等
		//页数
		int pageNumber = 1;

//		3、执行导出数据，导出数据核心代码要写到try当中，方便于出现错误时定位问题
		try {
			//遍历导出数据,分页导出数据到Excel，pageNumber < 5000是为了防止死循环
			while(pageNumber < 5000) {

				ExcelModelData<? extends BaseExcelModel> excelModelData =
						businessDataUtil.getExcelModelDataMap(pageNumber, 1000);
				//获得数据总条数
				Long totalCount = excelModelData.getTotalCount();
				//获得实际的导出列表数据
				List<? extends BaseExcelModel> writeDataList = excelModelData.getModelDataList();


				List<String> hideColumnList = businessDataUtil.getHideColumnList();
				if(ListUtil.isNullorEmpty(hideColumnList)) {
					excelWriteUtil.excelWrite(sheetName, writeDataList);
				} else {
					hideColumnList.add(0, "excelImportErrorMessage");
					excelWriteUtil.excelWrite(sheetName, writeDataList, hideColumnList.stream().toArray(String[]::new));
				}

				//数据全部读取完成跳出循环
				if(ListUtil.isNullorEmpty(writeDataList) || pageNumber * 1000 > totalCount) {
					break;
				}

				//页数+1
				pageNumber++;

			}

		} catch(Exception e) {
			//导出时业务出现异常，需要再导出的最后一行添加错误信息
			List<? extends BaseExcelModel> writeErrorDataList = new ArrayList<>();
			writeErrorDataList.add(businessDataUtil.getExcelErrorModelDataMap(e));
			excelWriteUtil.excelWrite(sheetName, writeErrorDataList);

			e.printStackTrace();
		} finally {
			if(excelWriteUtil != null) {
				//写入结束关闭流
				excelWriteUtil.finishWrite();
			}
		}

		ExportExcelVO exportExcelVO = new ExportExcelVO();
		exportExcelVO.setBusinessId(businessId);
		//相对resource目录全路径，示例：excelwritepath/2021/08/20/ae37d4bd-3095-4edb-9798-5db52d85da34.xlsx
		String excelWritePath = excelWriteUtil.getExcelWritePath();
		exportExcelVO.setExcelWritePath(excelWritePath);
		return exportExcelVO;
	}

	/**
	 * <p>追加后改文件名字这种方式不到万不得已不使用，先注释掉方法代码</p>
	 * @param filePath
	 * @param fileName
	 * @param sheetName
	 * @param data
	 * @param modelClass
	 * @author: ldc
	 * @version: 1.0
	 * @date 2022-5-9 14:59:40
	 */
//	@Deprecated
//	void excelWriter(String filePath, String fileName, String sheetName, Collection<?> data, Class modelClass) {
//		File templateFile = new File(filePath, fileName);
//		
//        File destFile = new File(filePath, UUID.randomUUID().toString() + fileName);
//        
//        // 创建ExcelWriter对象
//	    ExcelWriter excelWriter = null;
//        
//        try {
//            if (templateFile.exists()) {
//                //追加数据，目标文件与原始文件不能是同一个文件名
//                //withTemplate()指定模板文件
//                excelWriter = EasyExcel.write().withTemplate(templateFile)
//                    	//指定目标文件，不能与模板文件是同一个文件
//                        .file(destFile).autoCloseStream(false).build();
//            } else {
//                excelWriter = EasyExcel.write(templateFile, modelClass).build();
//            }
//            WriteSheet writeSheet = EasyExcel.writerSheet(sheetName).build();
//            excelWriter.write(data, writeSheet);
//        } finally {
//            // 千万别忘记finish 会帮忙关闭流
//            if (excelWriter != null) {
//                excelWriter.finish();
//            }
//        }
//
//        if (destFile.exists()) {
//            //删除原模板文件，新生成的文件变成新的模板文件
//            templateFile.delete();
//            destFile.renameTo(new File(filePath, fileName));
//        }
//	}
}
